//************************************************************************* //
// Game Engine
// Author: Szirmay-Kalos Laszlo, 2001. November.
//************************************************************************* //
#include <math.h>
#include <iostream.h>
#include <windows.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include "color.h"
#include "vector.h"
#include "ImageFile.h"
#include "glutwindow.h"

#define M_PI 3.1415
#define EPSILON 0.00001f

//=============================================================
class Ray {             // a sugar
//=============================================================
    Vector start;     // kezdopont
    Vector dir;       // normalizalt iranyvektor
public:
    Ray( Vector& start0, Vector& dir0 ) : start( start0 ), dir( dir0 ) { }
    Vector& Dir() { return dir; }
    Vector& Start() { return start; }
    Vector  GetPoint( float t ) { return ( start + dir * t); }
};
	
//=============================================================
class Sphere {        // a gomb
//=============================================================
    Vector center;     // kozeppent
    float  radius;     // sugar
public:
    Sphere( Vector& c, float r ) : center( c ) { radius = r; }

	BOOL Intersect( Ray& r, float& t ) {
		if ( radius <= EPSILON ) return FALSE;

		Vector dist = r.Start() - center;
		float b = (dist * r.Dir()) * 2.0;
		float a = (r.Dir() * r.Dir());
		float c = (dist * dist) - radius * radius;

		float discr = b * b - 4.0 * a * c;
		if ( discr < 0 ) return FALSE;
		float sqrt_discr = (float)sqrt( discr );
		float t1 = (-b + sqrt_discr)/2.0/a;
		float t2 = (-b - sqrt_discr)/2.0/a;

		if (t1 < EPSILON) t1 = -EPSILON;
		if (t2 < EPSILON) t2 = -EPSILON;
		if (t1 < 0 && t2 < 0) return FALSE;

		if ( t1 < 0 && t2 >= 0 )        t = t2;
		else if ( t2 < 0 && t1 >= 0 )   t = t1;
		else if (t1 < t2)               t = t1;
		else                            t = t2;
		return TRUE;	
	}
};


//--------------------------------------------
 class GameObject {
//--------------------------------------------
 protected:
	Vector position, velocity, acceleration;	// pozci, sebessg, gyorsuls
	BOOL   alive;								// letben van-e
    float  bounding_radius;						// befoglal gmb sugara
 public: 
	GameObject( Vector pos0 ) : position( pos0 ) { alive = TRUE; bounding_radius = 0; }

	BOOL IsAlive( ) { return alive; }

	virtual void Kill( ) { alive = FALSE; }

	virtual int GetType( ) = 0;					// tpus lekrdezs

	virtual void InteractIt( GameObject * obj ) { }

	virtual BOOL CollideIt( GameObject * obj, float& hit_time, Vector& hit_point ) {
		Sphere comb_sphere( obj->position, bounding_radius + obj->bounding_radius );
		Ray ray( position, velocity - obj->velocity );
		if ( comb_sphere.Intersect( ray, hit_time ) ) {
			Vector hit_pos = position + velocity * hit_time;
			Vector obj_hit_pos = obj->position + obj->velocity * hit_time;
			float a = bounding_radius / (bounding_radius + obj->bounding_radius);
			hit_point = hit_pos * (1-a) + obj_hit_pos * a;
			return TRUE;
		} else return FALSE;
	}	

	virtual void ControlIt( float dt ) { } 

	virtual void AnimateIt( float dt ) {
		if (alive) {
			position += velocity * dt + acceleration * (dt * dt / 2.0f);
			velocity += acceleration * dt;
		}
	}

	virtual void DrawIt( ) { }

	Vector& Position() { return position; } 
};

//--------------------------------------------
 class Member : public GameObject {
//--------------------------------------------
 protected:
  	 static Member * root;
	 Member	* next;
 public:
	Member( Vector pos0 ) : GameObject( pos0 ) { next = NULL; if ( !root ) root = this; }

	void Join( Member * obj ) { 
		if ( next ) next -> Join( obj ); 
		else	    next = obj;
	}

	void Interact( Member * obj ) {
		if ( obj != this  ) InteractIt( obj );
		if ( obj -> next )	Interact( obj -> next );
	}

	GameObject * Collide( Member * obj, float& mhit_time, Vector& mhit_point, GameObject * source ) {
		GameObject * hit_obj = NULL;
		if ( obj -> next ) hit_obj = Collide( obj -> next, mhit_time, mhit_point, source );

		float hit_time;
		Vector hit_point;

		if ( obj != this && obj != source && CollideIt( obj, hit_time, hit_point ) &&  hit_time < mhit_time ) {
			mhit_time = hit_time;
			mhit_point = hit_point;
			hit_obj = obj;
		} 
		return hit_obj;
	}

	void Control( float dt ) {
		ControlIt( dt );
		if ( next ) next -> Control( dt );
	}

	void Animate( float dt ) {
		AnimateIt( dt );
		if ( next )	next -> Animate( dt );
	}

	virtual void BeforeDraw( ) {
		glPushMatrix();
	}

	void Draw( ) {
		BeforeDraw( );
		DrawIt( );
		AfterDraw( );
		if ( next )	next -> Draw( );
	}
	virtual void AfterDraw( ) {
		glPopMatrix();
	}

	void BuryDead( Member * exclude ) {
		for( Member * m = root; m -> next != NULL; m = m -> next ) {
			if ( m -> next -> IsAlive() == FALSE && m -> next != exclude ) {
				Member * dead = m -> next;
				m -> next = m -> next -> next;
				delete dead;
				if ( m -> next == NULL ) return;
			}
		}
	}
};

//--------------------------------------------
class Avatar : public Member {
//--------------------------------------------
protected:
	Vector up;

public:
	Avatar( Vector& pos0 ) : Member( pos0 ), up( 0, 1, 0 ) {
		velocity = CVector( 0.0f, 0.0f, -0.1f ); 
		bounding_radius = 0.5f;
	}

	void	SetCameraTransform( ) {
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		gluLookAt( position.X(), position.Y(), position.Z(), 
				   position.X() + velocity.X(), position.Y() + velocity.Y(), position.Z() + velocity.Z(), 
				   up.X(), up.Y(), up.Z());
	}
	virtual void ProcessInput(GLUTWindow * input ) {}
};

//--------------------------------------------
class Texture {
//--------------------------------------------
	unsigned int texture_id;
public:
	Texture( char * filename );
	unsigned int Id() { return texture_id; }
};

//--------------------------------------------
class TexturedObject : public Member {
//--------------------------------------------
protected:
	Texture texture;
public:
	TexturedObject( Vector pos0, char * texture_filename ) 
		: Member( pos0 ), texture( texture_filename ) { }

	void BeforeDraw( ) {
		Member :: BeforeDraw();
		glEnable( GL_TEXTURE_2D );
		glBindTexture(GL_TEXTURE_2D, texture.Id());
     	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	}
	void AfterDraw( ) {
		glDisable( GL_TEXTURE_2D );
		Member :: AfterDraw();
	}
};

//--------------------------------------------
class BillBoard : public TexturedObject {
//--------------------------------------------
	float size;
public:
	BillBoard( Vector pos0, float size0, char * texture_filename ) 
		: TexturedObject( pos0, texture_filename ) { size = size0; }

	void  DrawIt();
};

#define PMRND   (2.0f * (float)rand()/RAND_MAX - 1.0f)

//--------------------------------------------
class Particle {
//--------------------------------------------
protected:
	Vector  position;			// current position of the particle
	Vector  prev_position;	// last position of the particle
	Vector  velocity;			// direction and speed
	Vector  acceleration;		// acceleration

	float   age;           // determines how long the particle is alive
	float	time_to_live;

	float   size;             // size of particle
	float   dsize;        // amount to change the size over time

	float   weight;           // determines how gravity affects the particle
	float   dweight;      // change over time

	Color   color;         // current color of the particle
	Color   dcolor;    // how the color changes with time

	Particle * next;

	Vector Randomize( Vector& mean, float variation ) { 
		Vector rand_vec(PMRND, PMRND, PMRND);
		return ( mean + rand_vec * variation ); 
	}
	Color  Randomize( Color& mean, Color& variation ) { return ( mean + variation * PMRND ); }
	float  Randomize( float mean, float variation ) { return ( mean + PMRND * variation ); }


public:
	Particle( ) { age = 0.0f; next = NULL; }

	void SetNext( Particle * p ) { next = p; }
	Particle * GetNext( ) { return next; }

	BOOL IsAlive( float dt ) { return (time_to_live > dt); }

	Particle * Animate( float dt ) {
		Particle * new_next = next;
		if ( next ) new_next = next -> Animate( dt );

		if (time_to_live > 0) {
			AnimateIt( dt );
			next = new_next;
			return this;
		} else { 
			delete this;
			return new_next;
		}
	}

	void AnimateIt( float dt ) {
		time_to_live -= dt;
		position += velocity * dt;
		velocity += acceleration * dt;

		size += dsize * dt;
		color += dcolor * dt;
	}

	void Draw( Vector& right, Vector& up ) {
		DrawIt( right, up );
		if ( next ) next -> Draw( right, up );
	}

	void DrawIt( Vector& right, Vector& up ) {
		if (time_to_live < 0) return;

		glColor4fv( color.GetArray() );
		glTexCoord2f(0.0, 0.0); glVertex3fv((position - (right + up) * size).GetArray() );
		glTexCoord2f(1.0, 0.0); glVertex3fv((position + (right - up) * size).GetArray() );
		glTexCoord2f(1.0, 1.0); glVertex3fv((position + (right + up) * size).GetArray() );
		glTexCoord2f(0.0, 1.0); glVertex3fv((position + (up - right) * size).GetArray() );
	}

	virtual ~Particle( ) { }
};


//--------------------------------------------
class ParticleSystem : public TexturedObject {
//--------------------------------------------
protected:
	Particle *	particles;    // particles for this emitter

	float       age;			// used to track how long since the last particle was emitted

	Vector		force;            // force (gravity, wind, etc.) acting on the particle system

public:
	ParticleSystem( Vector pos0, char * texture_filename ) 
		: TexturedObject( pos0, texture_filename ) {
		  particles = NULL;
		  age = 0.0;
		  alive = TRUE;
	}

	void  ControlIt(float dt) { 
		if (particles == NULL) alive = FALSE;
	}

	void  AnimateIt(float dt) { 
		if ( particles ) particles = particles -> Animate( dt ); 
	}

	void  DrawIt();

	virtual void Emit(int n) {}

	~ParticleSystem() { if (particles) delete particles; }
};

//--------------------------------------------
class GameEngine : public GLUTWindow {
//--------------------------------------------
 protected:
	Member		* world;
	Avatar		* avatar;
 public:
	GameEngine( int width, int height, char * caption ) { 
		
		glViewport(0, 0, width, height);		// reset the viewport to new dimensions

		glMatrixMode(GL_PROJECTION);			// set projection matrix current matrix
		glLoadIdentity();						// reset projection matrix

		gluPerspective( 90,					// field of view in angles, 
						(GLfloat)width/(GLfloat)height,					// aspect ratio
						0.1,				// znear
						100.0				// zfar
					  );

		glShadeModel(GL_SMOOTH);
	
		glEnable(GL_DEPTH_TEST);
		glEnable(GL_DITHER);
	}

	
	void Do_a_Step( float dt ) {
		avatar -> ProcessInput( this );

		world -> Animate( dt );
		world -> Control( dt );
		world -> BuryDead( avatar );

		glClearColor(0, 0, 0, 0);								// clear to black
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);		// clear screen and depth buffer

		avatar -> SetCameraTransform();
		world -> Draw();
		SwapBuffers( );
	}
};

