//************************************************************************* //
// Kill Game
// Author: Szirmay-Kalos Laszlo, 2001. November.
//************************************************************************* //
#include <math.h>
#include "GameEngine.h"

enum GameObjectType {
	SPACE,
	PLANET,
	SHIP,
	AVATAR,	
	EXPLOSION,
	BULLET
};

const float fNewton = 1.0f;

//============================================
// Space-game specific stuff
//============================================

#define SPACE_SIZE 20
//--------------------------------------------
 class Space : public TexturedObject {
//--------------------------------------------
 public:
	 Space( char * filename ) : TexturedObject( CVector( 0, 0, 0 ), filename ) { }

	 int GetType( ) { return SPACE; }

	 void DrawIt( ) {
		glBegin(GL_QUADS);
		// -z plane
		glTexCoord2i(0, 0); glVertex3i(-SPACE_SIZE, -SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(0, 1); glVertex3i(-SPACE_SIZE,  SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(1, 1); glVertex3i( SPACE_SIZE,  SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(1, 0); glVertex3i( SPACE_SIZE, -SPACE_SIZE, -SPACE_SIZE);

		// z plane
		glTexCoord2i(0, 0); glVertex3i(-SPACE_SIZE, -SPACE_SIZE, SPACE_SIZE);
		glTexCoord2i(0, 1); glVertex3i( SPACE_SIZE, -SPACE_SIZE, SPACE_SIZE);
		glTexCoord2i(1, 1); glVertex3i( SPACE_SIZE,  SPACE_SIZE, SPACE_SIZE);
		glTexCoord2i(1, 0); glVertex3i(-SPACE_SIZE,  SPACE_SIZE, SPACE_SIZE);

		// x plane
		glTexCoord2i(0, 0); glVertex3i( SPACE_SIZE, -SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(0, 1); glVertex3i( SPACE_SIZE, -SPACE_SIZE,  SPACE_SIZE);
		glTexCoord2i(1, 1); glVertex3i( SPACE_SIZE,  SPACE_SIZE,  SPACE_SIZE);
		glTexCoord2i(1, 0); glVertex3i( SPACE_SIZE,  SPACE_SIZE, -SPACE_SIZE);

		// -x plane
		glTexCoord2i(0, 0); glVertex3i(-SPACE_SIZE, -SPACE_SIZE,  SPACE_SIZE);
		glTexCoord2i(0, 1); glVertex3i(-SPACE_SIZE, -SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(1, 1); glVertex3i(-SPACE_SIZE,  SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(1, 0); glVertex3i(-SPACE_SIZE,  SPACE_SIZE,  SPACE_SIZE);

		// y plane, or top
		glTexCoord2i(0, 0); glVertex3i(-SPACE_SIZE,  SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(0, 1); glVertex3i( SPACE_SIZE,  SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(1, 1); glVertex3i( SPACE_SIZE,  SPACE_SIZE,  SPACE_SIZE);
		glTexCoord2i(1, 0); glVertex3i(-SPACE_SIZE,  SPACE_SIZE,  SPACE_SIZE);

		// -y plane, or bottom
		glTexCoord2i(0, 0); glVertex3i(-SPACE_SIZE, -SPACE_SIZE, -SPACE_SIZE);
		glTexCoord2i(0, 1); glVertex3i(-SPACE_SIZE, -SPACE_SIZE,  SPACE_SIZE);
		glTexCoord2i(1, 1); glVertex3i( SPACE_SIZE, -SPACE_SIZE,  SPACE_SIZE);
		glTexCoord2i(1, 0); glVertex3i( SPACE_SIZE, -SPACE_SIZE, -SPACE_SIZE);

		glEnd();
	} 
};

//--------------------------------------------
 class Planet : public TexturedObject {
//--------------------------------------------
	GLUquadricObj * quadric;
	float	mass;
	float	rot_angle, rot_speed;
	float   rev_angle, rev_speed;
    float   dist;
public:
	Planet( Vector& pos0, float R, char * filename ) : TexturedObject( pos0, filename ) {
		bounding_radius = R;
		mass = R * R * R;
		dist = pos0.Length();
		rot_angle = rev_angle = 0.0f;
		rev_speed = (dist > EPSILON) ? 100/dist/dist/dist : 0;
		rot_speed = 5.0;

		quadric = gluNewQuadric();
		gluQuadricTexture(quadric, GL_TRUE);
	}
	
	int GetType( ) { return PLANET; }
	float Radius( ) { return bounding_radius; }
	float Mass( ) { return mass; }
	
	void DrawIt( ) { 
		glRotatef( rev_angle, 0, 0, 1);
		glTranslatef( dist, 0, 0 ); 
		glRotatef( rot_angle, 0, 0, 1 );
		gluSphere( quadric, bounding_radius, 16, 10 );
	}

	void AnimateIt( float dt ) {
		rot_angle += rot_speed * dt;
		if (rot_angle > 360.0f) rot_angle -= 360.0;
		rev_angle += rev_speed * dt;
		if (rev_angle > 360.0f) rev_angle -= 360.0;
		position.X() = dist * cos( rev_angle * M_PI/180 );
		position.Y() = dist * sin( rev_angle * M_PI/180 );
		position.Z() = 0;
	}
};

//--------------------------------------------
class ExplosionParticle : public Particle {
//--------------------------------------------
public:
	ExplosionParticle( Vector& center, float spread ) {
		position = Randomize( center, 0.1f );
		time_to_live = Randomize( 2.0f, 1.0f );
		
		size = 0.001f; ;
		dsize = Randomize( 1.0f, 0.5f ) / time_to_live / 2.0; 
		weight = Randomize( 1.0, 0.1f );
		velocity = Randomize( CVector(0.0f, 0.0f, 0.0f), 0.4f );
		acceleration = Randomize( CVector(0.0f, 0.0f, 0.0f), 0.4f );
		color = Randomize( Color(1.0f, 0.5, 0.0f, 1.0f), Color(0.0f, 0.5f, 0.0f, 0.0f) );
		dcolor = Color(0.0f, -0.5f, 0.0f, -1.0f) / time_to_live / 2.0f;
	}
};

//--------------------------------------------
class Explosion : public ParticleSystem {
//--------------------------------------------
public:
	Explosion( Vector pos0 ) : ParticleSystem( pos0, "explosion.bmp" ) { 
		Emit( 100 ); 
	}

	int GetType( ) { return EXPLOSION; }

	void Emit(int n) {
		for( int i = 0; i < n; i++ ) {
			ExplosionParticle * particle = new ExplosionParticle( position, 0.1f );
			particle -> SetNext( particles );
			particles = particle;
		}
	}
};

//--------------------------------------------
 class Bullet : public BillBoard {
//--------------------------------------------
	float	age;
	GameObject * source;
 public:
	Bullet( Vector& pos0, Vector& v, GameObject * s ) 
		: BillBoard( pos0, 0.3f, "photon.bmp" ) { 
		velocity = v * 3;
		age = 0;
		source = s;
	}

	int GetType( ) { return BULLET; }

	void ControlIt( float dt ) {	// process interaction, AI, etc.
		float hit_time = dt;
		Vector hit_point;
		GameObject * hit_object = Collide(root, hit_time, hit_point, source); 

		if ( hit_object ) {
			Kill();
			int hit_object_type = hit_object -> GetType();
			if ( hit_object_type != AVATAR )  root -> Join( new Explosion( hit_point ) );
			if ( hit_object_type != PLANET ) hit_object -> Kill();
		}
		age += dt;
		if ( age > 10 ) Kill();
	}
};

//--------------------------------------------
 class Ship : public TexturedObject {
//--------------------------------------------
	float	mass;
	Vector gravity_force, rocket_force;
	enum AI_State {
		ESCAPE_FROM_PLANET,
		CHASE_AVATAR,
	} state;
	float last_shot;
	struct Vector3 { float x, y, z; } * shipgeom;
	struct Vector2 { float u, v;} * shiptext;
	int nvertex;
 public:
	 Ship( Vector& pos0 ) : TexturedObject( pos0, "ship_texture.bmp" ) {
		mass = 0.1f;
		velocity = CVector( 0.1f, 0.2f, 0.3f );
		state = CHASE_AVATAR;
		last_shot = 0;

		FILE * fgeom = fopen("ship.geo","r");
		if ( fgeom == NULL ) {
			cerr << "ship.geo not found\n";
			exit( 0 );
		}
		fscanf(fgeom,"%d", &nvertex);
		shipgeom = new Vector3[nvertex];
		float r2 = 0;
		for(int i = 0; i < nvertex; i++) { 
			fscanf(fgeom, "%f %f %f\n", &shipgeom[i].x, &shipgeom[i].y, &shipgeom[i].z);
			float rn2 = shipgeom[i].x * shipgeom[i].x + shipgeom[i].y * shipgeom[i].y + shipgeom[i].z * shipgeom[i].z;
			if (rn2 > r2) r2 = rn2;
		}
		bounding_radius = sqrt(r2);
		fclose(fgeom);

		FILE * ftext = fopen("ship.tex","r");
		if ( ftext == NULL ) {
			cerr << "ship.tex not found\n";
			exit( 0 );
		}
		fscanf(ftext,"%d", &nvertex);
		shiptext = new Vector2[nvertex];
		for(i = 0; i < nvertex; i++) 
			fscanf(ftext, "%f %f\n", &shiptext[i].u, &shiptext[i].v);
		fclose(ftext);			
	}

	int GetType( ) { return SHIP; }

	void ControlIt( float dt ) {	// process interaction, AI, etc.
		if ( !alive ) return;
		float v = velocity.Length();
		if (v > 1) velocity *= 1/v;

		gravity_force = CVector(0, 0, 0);
		rocket_force = CVector(0, 0, 0);

		Interact( root );

		acceleration = (gravity_force + rocket_force) / mass;
		last_shot += dt;
	}
	
	void InteractIt( GameObject * object ) {
		if ( object -> GetType( ) == PLANET ) {
			Planet * planet = (Planet *)object;
			Vector dr = planet -> Position() - Position();
			float dist = dr.Length();

			gravity_force += dr * fNewton * mass * planet -> Mass() / dist / dist / dist;

			if (dist < planet -> Radius()) {
				Kill();
				root -> Join( new Explosion( position ) );
			}
			if ( dist < planet -> Radius() * 3 ) {
				rocket_force = (Position() - planet -> Position()) / dist;
				state = ESCAPE_FROM_PLANET;
			}		
			if ( dist > planet -> Radius() * 4 ) 
				state = CHASE_AVATAR;
		}	

		if ( object -> GetType( ) == AVATAR ) {
			Avatar * avatar = (Avatar *)object;
			Vector goal = avatar -> Position() - Position();
			float dist = goal.Length();
			goal *= 1/dist;
			Vector current = velocity;
			velocity.Normalize();

			if (state == CHASE_AVATAR) rocket_force = goal * (dist * 0.005);

			if ( last_shot > 2 && velocity * goal > 0.9 && dist < 6 )  {
				root -> Join( new Bullet(position, velocity, this) );
				last_shot = 0;
			}
		}	
	}

	void DrawIt() {
		glTranslatef( position.X(), position.Y(), position.Z() );
		Vector modell_head( 0.0f, 0.0f, 1.0f );
		Vector world_head = velocity.UnitVector();
		Vector rotate_axis = world_head % modell_head;
		
		float  cos_rotate_angle = world_head * modell_head;
		glRotatef( -acos( cos_rotate_angle ) * 180.0f / M_PI, rotate_axis.X(), rotate_axis.Y(), rotate_axis.Z() );

		glBegin( GL_QUADS );
			for(int i = 0; i < nvertex; i++) {
				glTexCoord2f( shiptext[i].u, shiptext[i].v );
				glVertex3f( shipgeom[i].x, shipgeom[i].y, shipgeom[i].z );
			}
		glEnd( );
	}
	
	void Kill( ) {
		cerr << "Ship is destroyed\n ";
		Member :: Kill();
	}
};

 
//--------------------------------------------
 class Self : public Avatar {
//--------------------------------------------
	int		lifes;
	float	mass;
	Vector	force, rocketForce;
 public:
	Self( Vector& pos0 ) : Avatar( pos0 ) { 
		mass = 0.001f;
		lifes = 5;
		bounding_radius = 0.2f;
	}

	int GetType( ) { return AVATAR; }

	void ControlIt( float dt ) {	// process interaction, AI, etc.
		if ( !alive ) return;

		float v = velocity.Length();
		if (v > 1) velocity *= 1/v;

		force = CVector(0.0f, 0.0f, 0.0f);

		Interact( root );

		acceleration = (force + rocketForce) / mass;
	}

	void InteractIt( GameObject * object ) {
		if ( object -> GetType( ) == PLANET ) {
			Planet * planet = (Planet *)object;
			Vector dr = planet -> Position() - Position();
			float distance = dr.Length();

			if (distance < planet -> Radius() ) Kill();
			force += dr * fNewton * mass * planet->Mass() / distance / distance / distance;
		}	
	}
	
	void Message(float x, float y, char * buffer) { 			
		glRasterPos2f( x, y );
		for( int i = 0; i < 80; i++ ) {
			if ( buffer[i] == '\0' ) break;
			else glutBitmapCharacter( GLUT_BITMAP_8_BY_13, buffer[i] );
		}
	}

	void DrawIt( ) {
		glMatrixMode( GL_MODELVIEW );
		glPushMatrix();
		glLoadIdentity();

		glMatrixMode( GL_PROJECTION );
		glPushMatrix();
		glLoadIdentity();
		gluOrtho2D(0.0, 1.0, 0.0, 1.0); 

		if ( alive ) {
			glDisable( GL_DEPTH_TEST );
			glColor3f( 1.0f, 1.0f, 1.0f );
			glBegin( GL_LINES );
			glVertex2f(0.45f, 0.5f); 
			glVertex2f(0.55f, 0.5f); 
			glVertex2f(0.5f, 0.45f); 
			glVertex2f(0.5f, 0.55f); 
			glEnd();
			char buffer[80];
			glEnable( GL_DEPTH_TEST );
			sprintf(buffer, "%1.0f, %1.0f, %1.0f", position.X(), position.Y(), position.Z());
			Message(0.55f, 0.55f, buffer);
		} else {
			Message( 0.25f, 0.60f, "You have been killed");
			char buffer[80];
			sprintf(buffer, "You still have %d lifes.", lifes);
			Message( 0.25f, 0.55f, buffer);
			Message( 0.25f, 0.50f, "Press 's' to continue!");
		}
		glPopMatrix( );
		glMatrixMode( GL_MODELVIEW );
		glPopMatrix();
	}

	void ProcessInput( GLUTWindow * input ) {
		if ( alive ) {
			rocketForce = CVector(0.0f, 0.0f, 0.0f);

			Vector head = velocity.UnitVector();

			if ( input->GetKeyStatus('w') )  {
				velocity *= -1;
			}
			if ( input->GetKeyStatus('q') )  rocketForce = head;
			if ( input->GetKeyStatus('a') )  rocketForce = head * (-1.0f);

			if ( input->GetKeyStatus(' ') ) 
				root -> Join( new Bullet(position + velocity, velocity, this) );

			if ( input->IsUp() )	rocketForce = up * (-1.0f);
			if ( input->IsDown() )  rocketForce = up;
			if ( input->IsLeft() )  rocketForce = up % head;
			if ( input->IsRight() ) rocketForce = head % up;
		
			rocketForce *= 0.0001f;
		} else {
			if ( input->GetKeyStatus('s') ) {
				if ( lifes > 0 ) {
					position = CVector(0.0f, 0.0f, 9.0f);
					velocity = CVector(0, 0, 0);
					alive = TRUE;
				}
			}
		}
	}

	void Kill() {
		if ( alive ) {
			alive = false;
			lifes--;
		}
	}
 };
//--------------------------------------------
 class KillGame : public GameEngine {
//--------------------------------------------
 public:
	 KillGame( int width, int height, char * caption ) : GameEngine( width, height, caption ) {

		world = new Space( "stars.bmp" );

		world -> Join( new Ship( CVector(0.0f, 6.0f, 0.0f) ) );
		world -> Join( new Ship( CVector(-3.0f, -2.0f, 0.0f) ) );
		world -> Join( new Ship( CVector(5.0f, -6.0f, 0.0f) ) );
		world -> Join( new Ship( CVector(6.0f, 0.0f, -2.0f) ) );
		world -> Join( new Ship( CVector(5.0f, -5.0f, -3.0f) ) );
		world -> Join( new Ship( CVector(-6.0f, 4.0f, -4.0f) ) );

		world -> Join( new Planet( CVector(0.0f, 0.0f, 0.0f), 1.0f, "fire.bmp" ) );
		world -> Join( new Planet( CVector(3.0f, 0.0f, 0.0f), 0.4f, "neptun.bmp" ) );
		world -> Join( new Planet( CVector(-5.0f, 0.0f, 0.0f), 0.4f, "earth.bmp" ) );
		world -> Join( new Planet( CVector(-6.0f, 0.0f, 0.0f), 0.1f, "moon.bmp" ) );
		world -> Join( new Planet( CVector(7.0f, 0.0f, 0.0f), 0.4f, "mars.bmp") );
		world -> Join( new Planet( CVector(-9.0f, 0.0f, 0.0f), 0.4f, "neptun.bmp" ) );

		avatar = new Self( CVector(0.0f, 0.0f, 9.0f) );
		world -> Join( avatar );
	}
};

void Application::Start( ) {
	new KillGame( 600, 600, "SpaceGame" );
}