#include <stdio.h>
#include <stdlib.h>
#include <ADuC812.h>

#define RESET_WINDOW P3_3
#define SENSE_LED P3_6
#define LOCK_LED P3_7

#define BIGSTEPSIZE 100 // *100ps
#define SMALLSTEPSIZE 5 // *100ps
#define BIG_SMALL_RATIO (BIGSTEPSIZE/SMALLSTEPSIZE)
#define MAXBUFLEN 16    // must be power of 2
#define NO_AVERAGE 1
#define SMALL_AVERAGE 4
#define FULL_AVERAGE 10

unsigned char proctype=0x00; /* 0x00=812, 0x01=814, 0x02=816, 0x03=824, 0x10=841, 0x11=842, 0x12=843 */
unsigned char dly_cycles=225;
long clk=11059200;

unsigned char Irq0Executed;
unsigned char bigstep, smallstep, window;
unsigned char locked, mid_window;
unsigned char C_bigLow, C_3Low, C_mid, C_3High, C_bigHigh;

unsigned char winbuf[MAXBUFLEN];
unsigned char winbuf_i;
unsigned char winbuf_n, winbuf_nmax;

void Delay_ms(short ms)
{
	short i;
	for(i=0; i<ms; i++)
	{
		{ /* about 1ms, 11059200/12 Hz */
			_asm
				push b
				mov b,_dly_cycles
			00001$:
				djnz b,00002$
				mov b,_dly_cycles
			00002$:
				djnz b,00002$
				pop b
			_endasm;
		}
	}
}

char SIn(void)
{
	while (!RI);
	RI=0;
	return SBUF;
}

void SOut(char a)
{
	while (!TI);
	TI=0;
    SBUF=a;
}

void OutString(char *s)
{
	unsigned char i;

	i=0;
	while (i<127 && s[i]) SOut(s[i++]);
}

void SInit(long baud_rate, char mode)
{
	if (mode==2)
	{
		unsigned short u;
		u=65536L-(2*((proctype<0x10)? clk/2:clk)+16*baud_rate)/(2*16*baud_rate);
		RCAP2H=u>>8;
		RCAP2L=u;
		TH2=u>>8;
		TL2=u;
		SCON=0x52;
		T2CON=0x34;
	}
	else
	{
		TH1=256-(2*((proctype<0x10)? clk/12:clk)+16*baud_rate)/(2*16*baud_rate);
		PCON|=SMOD;	// set double baud rate
		TMOD=(TMOD & T0_MASK) | 0x20;	// timer 1 mode : 8-bit auto reload
		SCON=0x50;	// Set Serial for mode 1 & enable reception
		TR1=1;		// Start timer 1
	}
    TI=1;          /* ti is normally set in this program */
    RI=0;          /* ri is normally cleared */
}

/*****************************************
This interrupt routine is executed on every trigger event
******************************************/

void IrqHandler(void) interrupt IE0_VECTOR
{
	unsigned char i,w;
	short sum;

	Irq0Executed=1; // trigger signal detected
	/* 
	the laser pulse can be detected in 5 time windows,
	 0: 0..t0
	 1: t0..to+6ns
	 3: t0+6ns..t0+12ns
	 7: t0+12ns..t0+15ns
	15: after t0+15ns
	*/
	w=P1 & 0x0F;  // which time window includes the pulse?
	if (w==0) window=0;
	else if (w==1) window=1;
	else if (w==3) window=2;
	else if (w==7) window=3;
	else if (w==15) window=4;
	else
	{
		window=255;			// laser pulse not detected
		SENSE_LED=1;  		// switch off LED
		LOCK_LED=1; 	 	// not locked
		RESET_WINDOW=0;		// reset the window detector D flip-flops
		RESET_WINDOW=1;
	}
	if (window < 5)
	{
	    winbuf_i=(winbuf_i+1) & (MAXBUFLEN-1);
	    winbuf[winbuf_i] = window;
	    if (winbuf_n < winbuf_nmax) winbuf_n++;
	    else winbuf_n = winbuf_nmax;
	    sum = 0;
	    for (i = 0; i < winbuf_n; i++)
	    {
	        sum = sum + 10 * winbuf[(winbuf_i-i) & (MAXBUFLEN-1)];
	    }
	    window = sum / winbuf_n; 		// averaged window value * 10 (range is : 0..40)
		mid_window = 8;					// half width of the middle, locked range
		if (locked > 1) mid_window =5; 	// it is dinamically reduced
		locked=0;						// assume not locked
		if (window <= C_bigLow)			// big step up
		{
			if (bigstep<255) bigstep++;
		}
		else if (window <  C_3Low)		// locked range ?
		{
			locked = 1;					// locked, but 3 small steps needed
			if (smallstep<252) smallstep= smallstep + 3;
			else if (bigstep<255)
			{
				smallstep-=BIG_SMALL_RATIO;
				bigstep++;
				smallstep=smallstep + 3;
			}
		}
		else if (window < C_mid - mid_window) 	
		{
			locked = 1;							// locked, but small steps needed
			if (smallstep<255) smallstep++;
			else if (bigstep<255)
			{
				smallstep-=BIG_SMALL_RATIO;
				bigstep++;
				smallstep++;
			}
		}
		else if (window <= C_mid + mid_window)	// locked, no steps needed
		{
			locked=2;
		}
		else if (window< C_3High)				// locked, but small steps needed
		{
			locked = 1;
			if (smallstep>0) smallstep--;
			else if (bigstep>0)
			{
				smallstep+=BIG_SMALL_RATIO;
				bigstep--;
				smallstep--;
			}
		}
		else if (window< C_bigHigh)
		{
			locked = 1;							// locked, but 3 small steps needed
			if (smallstep>3) smallstep = smallstep - 3;
			else if (bigstep>0)
			{
				smallstep+=BIG_SMALL_RATIO;
				bigstep--;
				smallstep = smallstep - 3;
			}
		}
		else									// not locked, big steps needed
		{
			if (bigstep>0) bigstep--;
		}
		SENSE_LED=0;  		// laser pulse detected
		LOCK_LED=(!locked); // locked
		if (!locked)
		{
			winbuf_nmax=NO_AVERAGE;
		}
		else if (locked==1)
		{
			winbuf_nmax=SMALL_AVERAGE;
		}
		else
		{
			winbuf_nmax=FULL_AVERAGE;
		}
		P0=bigstep;			// send to hardware
		P2=smallstep;		// send to hardware
		RESET_WINDOW=0;		// reset the window detector D flip-flops
		RESET_WINDOW=1;
	}
}

void SetupSystem(void)
{
	Irq0Executed=0;		// we don't have any trigger events yet
	bigstep=128;	    // initial value
	smallstep=128;		// initial value
	C_bigLow = 1;       // threshold values for big step up
	C_3Low = 8;			// 3 small step up
	C_mid = 20;         // locked range
	C_3High = 32;       // 3 small step down
	C_bigHigh = 39;		// big step down
	winbuf_i=0;
	winbuf_n=0;
	winbuf_nmax=NO_AVERAGE;
	P0=bigstep;			// send to hardware
	P2=smallstep;		// send to hardware
	P1=0;				// the P1 port is a digital input port
	RESET_WINDOW=0;		// reset the window detector D flip-flops
	RESET_WINDOW=1;
	SENSE_LED=1;		// sense LED off
	LOCK_LED=1;			// lock LED off
	IT0=1;  			// edge sensitive interrupts
	EX0=1;  			// enable external interrupt
	EA=1;   			// enable global interrupts
}

void main(void)
{
	SInit(57600,2);  // setup serial communications
	OutString("Active delay control, 2006.03.14.\n");
	OutString("NRG, www.noise.physx.u-szeged.hu\n");
	SetupSystem();
	while (1)
	{
		while (!Irq0Executed);  // wait for a trigger event
		SOut('#');              // send data over the serial port
		SOut(bigstep);
		SOut(smallstep);
		SOut(window);
		Irq0Executed=0;
	}
}
