﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2010.02.03.
 * Idő: 10:38
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Drawing;
using System.Xml;
//
using Noise.DataAcquisition;
using TetheredSun.Core;
using TetheredSun.SignalAnalysis;

namespace GenericMonitor
{
	/// <summary>
	/// Description of ChannelSettings.
	/// </summary>
	public class ChannelSettings : IXmlConvertible
	{
		protected string name;
		protected bool isActive;
		protected bool isAutoscaled = false;
		protected LevelCrossingDetector levelCrossingDetector = null;
		protected Sensor sensor;
		protected double yMax = 5.0;			// The maximum value on the y axis of the plot.
		protected double yMin = 0.0;			// The minimum value on the y axis of the plot.
		protected double objectLength = 0.1;	// The length of the object that passes before the photogate.
		protected Color colour = Color.DarkSeaGreen;
		
		
		public ChannelSettings(string name, bool isActive, Sensor sensor)
		{
			this.name = name;
			this.isActive = isActive;
			this.sensor = sensor;
		}
		
		public ChannelSettings() : this(String.Empty, false, null) { }
		
		
		public Color Colour {
			get { return colour; }
			set { colour = value; }
		}
		
		
		public string Name {
			get { return name; }
			set { name = value; }
		}
		
		public bool IsActive {
			get { return isActive; }
			set { isActive = value; }
		}
		
		public bool IsAutoscaled {
			get { return isAutoscaled; }
			set { isAutoscaled = value; }
		}
		
		public LevelCrossingDetector LevelCrossingDetector {
			get { return levelCrossingDetector; }
			set { levelCrossingDetector = value; }
		}
		
		public Sensor Sensor {
			get { return sensor; }
			set { sensor = value; }
		}
		
		public double YMax {
			get { return yMax; }
			set { yMax = value; }
		}
		
		public double YMin {
			get { return yMin; }
			set { yMin = value; }
		}
		
		public double ObjectLength {
			get { return objectLength; }
			set { objectLength = value; }
		}
		
		
		public void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "ChannelSettings") {
				Exception exception = new ArgumentException("Invalid element name.");
				
				exception.Data.Add("Expected name", "ChannelSettings");
				exception.Data.Add("Received name", xmlElement.Name);
				
				throw exception;
			}
			
			XmlNode child;
			bool parsedBool;
			int parsedInt;
			double parsedDouble;
			string type;
			
			child = xmlElement.SelectSingleNode("child::Name");
			name = child.InnerXml;
			
			child = xmlElement.SelectSingleNode("child::IsActive");
			isActive = Boolean.TryParse(child.InnerXml, out parsedBool) ? parsedBool : true;
			
			child = xmlElement.SelectSingleNode("child::IsAutoscaled");
			isAutoscaled = Boolean.TryParse(child.InnerXml, out parsedBool) ? parsedBool : true;
			
			child = xmlElement.SelectSingleNode("child::YMin");
			yMin = Double.TryParse(child.InnerXml, out parsedDouble) ? parsedDouble : 0.0;
			
			child = xmlElement.SelectSingleNode("child::YMax");
			yMax = Double.TryParse(child.InnerXml, out parsedDouble) ? parsedDouble : 0.5;
			
			child = xmlElement.SelectSingleNode("child::Colour");
			if (Int32.TryParse(child.InnerXml, out parsedInt)) {
				colour = Color.FromArgb(parsedInt);
			} else {
				colour = Color.DarkSeaGreen;
			}
			
			child = xmlElement.SelectSingleNode("child::Sensor");
			if (child != null) {
				type = child.Attributes["type"].Value;
				
				switch (type) {
					case "Linear":
						sensor = new LinearSensor();
						break;
					case "Thermistor":
						sensor = new Thermistor();
						break;
					default:
						sensor = null;
						break;
				}
				
				if (sensor != null) sensor.FromXml(child as XmlElement);
			} else {
				sensor = null;
			}
			
			child = xmlElement.SelectSingleNode("child::ObjectLength");
			if (child != null) {
				objectLength = Double.TryParse(child.InnerXml, out parsedDouble) ? parsedDouble : 0.1;
			} else {
				objectLength = 0.1;
			}
			
			child = xmlElement.SelectSingleNode("child::LevelCrossingDetector");
			if (child != null) {
				levelCrossingDetector = new LevelCrossingDetector(0.0);
				levelCrossingDetector.FromXml(child as XmlElement);
			} else {
				levelCrossingDetector = null;
			}
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement channelSettingsXml = context.CreateElement("ChannelSettings");
			XmlElement nameXml = context.CreateElement("Name");
			XmlElement isActiveXml = context.CreateElement("IsActive");
			XmlElement isAutoScaledXml = context.CreateElement("IsAutoscaled");
			XmlElement yMinXml = context.CreateElement("YMin");
			XmlElement yMaxXml = context.CreateElement("YMax");
			XmlElement objectLengthXml = context.CreateElement("ObjectLength");
			XmlElement colourXml = context.CreateElement("Colour");
			
			nameXml.InnerXml = name;
			channelSettingsXml.AppendChild(nameXml);
			
			isActiveXml.InnerXml = isActive.ToString();
			channelSettingsXml.AppendChild(isActiveXml);
			
			isAutoScaledXml.InnerXml = isAutoscaled.ToString();
			channelSettingsXml.AppendChild(isAutoScaledXml);
			
			yMinXml.InnerXml = yMin.ToString();
			channelSettingsXml.AppendChild(yMinXml);
			
			yMaxXml.InnerXml = yMax.ToString();
			channelSettingsXml.AppendChild(yMaxXml);
			
			colourXml.InnerXml = colour.ToArgb().ToString();
			colourXml.SetAttribute("description", colour.ToString().Replace("Color", "Colour"));
			channelSettingsXml.AppendChild(colourXml);
			
			if (sensor != null) {
				channelSettingsXml.AppendChild(sensor.ToXml(context));
			}
			
			if (levelCrossingDetector != null) {
				objectLengthXml.InnerXml = objectLength.ToString();
				channelSettingsXml.AppendChild(objectLengthXml);
				channelSettingsXml.AppendChild(levelCrossingDetector.ToXml(context));
			}
			
			return channelSettingsXml;
		}
		
	}
}
