﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.11.20.
 * Idő: 13:32
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Drawing.Text;
using System.IO;
using System.IO.Ports;
using System.Reflection;
using System.Windows.Forms;
using System.Windows.Forms.DataVisualization;
using System.Windows.Forms.DataVisualization.Charting;
//
using BSE.Windows.Forms;
//
using Noise.Controls;
using Noise.DataAcquisition;
using TetheredSun.Core;
using TetheredSun.SignalAnalysis;

namespace GenericMonitor
{
	/// <summary>
	/// Description of MainForm.
	/// </summary>
	public partial class MainForm : Form
	{
		#region FIELDS
		protected Ftdi ftdi;
		protected Edaq530 device;
		protected ChannelInfo channelAInfo = new ChannelInfo("A");
		protected ChannelInfo channelBInfo = new ChannelInfo("B");
		protected ChannelInfo channelCInfo = new ChannelInfo("C");
		protected Channel channelA;
		protected Channel channelB;
		protected Channel channelC;
		protected double dt;
		protected bool isSampling = false;
		protected bool textBoxesInitialised = false;
		protected int levelCrossingCapacity = 10000;
		protected int maxPoints = 10000;
		protected int gridViewRows = 16;
		protected string meterFormat = "f2";
		protected Measurement measurement = new Measurement();
		protected ChannelList channels = new ChannelList();
		protected System.Windows.Forms.Timer meterRefreshTimer = new System.Windows.Forms.Timer();
		#endregion
		
		#region CONSTRUCTORS
		public MainForm()
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			//
			channelA = new Channel(channelAInfo);
			channelB = new Channel(channelBInfo);
			channelC = new Channel(channelCInfo);
			//
			SetFtdi();
			InitialiseDevice();
			InitialiseForm();
			LoadMeasurement("LastSettings.xml");
			SetControls();
			GetVersion();
		}
		#endregion
		
		#region METHODS
		public void AddPanelExpandingEventHandlers()
		{
			foreach (XPanderPanel panel in xPanderPanelList1.XPanderPanels) {
				panel.PanelExpanding += HandlePanelExpandingEvents;
			}
		}
		
		public void CopyToClipBoard(DataGridView dataGridView)
		{
			string clipboardString = String.Empty;
			
			// Write the header:
			if (dataGridView == aDataGridView) {
				clipboardString += "Channel A\t";
				if (measurement[0].Sensor != null) {
					clipboardString += measurement[0].Sensor.GetLabel();
				} else {
					clipboardString += "U [V]";
				}
			} else if (dataGridView == bDataGridView) {
				clipboardString += "Channel B\t";
				if (measurement[1].Sensor != null) {
					clipboardString += measurement[1].Sensor.GetLabel();
				} else {
					clipboardString += "U [V]";
				}
			} else if (dataGridView == cDataGridView) {
				clipboardString += "Channel C\t";
				if (measurement[2].Sensor != null) {
					clipboardString += measurement[2].Sensor.GetLabel();
				} else {
					clipboardString += "U [V]";
				}
			}
			
			clipboardString += "\t" + Environment.NewLine;
			
			for (int i = 0; i < dataGridView.ColumnCount; i++) {
				clipboardString += dataGridView.Columns[i].HeaderText + "\t";
			}
			
			clipboardString = clipboardString.Trim("\t".ToCharArray());
			clipboardString += Environment.NewLine;
			
			// Write the rest of the data:
			for (int i = 0; i < dataGridView.RowCount - 1; i++) {
				for (int j = 0; j < dataGridView.ColumnCount; j++) {
					clipboardString += (dataGridView.Rows[i].Cells[j].Value != null) ?
						dataGridView.Rows[i].Cells[j].Value.ToString() : String.Empty;
					clipboardString += "\t";
				}
				clipboardString = clipboardString.Trim("\t".ToCharArray());
				clipboardString += Environment.NewLine;
			}
			

	
			Clipboard.SetDataObject(clipboardString);
		}
		
		public void GetSamplingParameters()
		{
			double? fs = device.SamplingFrequency;
			
			measurement.Averages = device.Averages;
			if (measurement.Averages == null) measurement.Averages = Averages.One;
			measurement.SamplingFrequency = fs.HasValue ? fs.Value : 100.0;
			dt = 1.0 / measurement.SamplingFrequency;
			//
			channelA.TimeIncrement = dt;
			channelB.TimeIncrement = dt;
			channelC.TimeIncrement = dt;
			//
			fsNumericControl.SetValue(measurement.SamplingFrequency);
			//
			averagesComboBox.SelectedIndex = measurement.Averages.Code;
			//
			xWidthNumericControl.SetValue(measurement.XScalePoints * dt);
		}
		
		public void GetVersion()
		{
			AssemblyName assemblyName = Assembly.GetExecutingAssembly().GetName();
			
			versionLabel.Text = String.Format("Version: {0}.{1}{2}{3}", assemblyName.Version.Major, assemblyName.Version.Minor,
			                                 assemblyName.Version.Build, assemblyName.Version.Revision);
		}
		
		public void InitialiseDevice()
		{
			if (ftdi == null) {
				device = null;
				fsNumericControl.Enabled = false;
				averagesComboBox.Enabled = false;
				startToolStripButton.Enabled = false;
				return;
			}
			
			
			device = new Edaq530(ftdi);
			device.Reset();
			device.Channels = channels;
			device.Channels.Add(channelA);
			device.Channels.Add(channelB);
			device.Channels.Add(channelC);
			device.StoppedOnError += delegate { Stop(); };
			//
			toolStripStatusLabel1.Text = device.GetDeviceInfo();
			//
			fsNumericControl.Enabled = true;
			averagesComboBox.Enabled = true;
			startToolStripButton.Enabled = true;
		}
				
		public void InitialiseForm()
		{
			AddPanelExpandingEventHandlers();
			
			aSensorControl.SensorRecreated += HandleSensorRecreatedEvents;
			aSensorControl.SensorUpdated += HandleSensorUpdatedEvents;
			bSensorControl.SensorRecreated += HandleSensorRecreatedEvents;
			bSensorControl.SensorUpdated += HandleSensorUpdatedEvents;
			cSensorControl.SensorRecreated += HandleSensorRecreatedEvents;
			cSensorControl.SensorUpdated += HandleSensorUpdatedEvents;
			//
			xPointsNumericControl.MaxValue = maxPoints;
			//
			meterRefreshTimer.Interval = 500;
			meterRefreshTimer.Tick += RefreshMeters;
			//
			measurement[0].Colour = Color.DarkCyan;
			measurement[1].Colour = Color.Maroon;
			measurement[2].Colour = Color.RoyalBlue;
			//
			aDataPlotter.Channel = channelA;
			bDataPlotter.Channel = channelB;
			cDataPlotter.Channel = channelC;
			//
			aDataGridView.Rows.Clear();
			bDataGridView.Rows.Clear();
			cDataGridView.Rows.Clear();
		}
		
		public void LoadMeasurement(string path)
		{
			try {
				measurement = Measurement.Open(path);
			} catch (Exception exception) {
				ExceptionReporter.Show(exception, true);
				measurement = new Measurement();
				measurement[0].Colour = Color.DarkCyan;
				measurement[1].Colour = Color.Maroon;
				measurement[2].Colour = Color.RoyalBlue;
			} finally {
				SetControls();
			}
		}
		
		public void RefreshMeters(object sender, EventArgs e)
		{
			if (sender == meterRefreshTimer) {
				if (channelA != null) {
					aTextBox.Text = (channelA.ChannelInfo.Sensor != null) ?
						channelA.ChannelInfo.Sensor.GetValueString(channelA.CurrentValue, meterFormat, true) :
						String.Format("U = {0} V", channelA.CurrentValue.ToString(meterFormat));
					                                                                                                           
				} else {
					aTextBox.Text = "NO DATA";
				}
				//
				if (channelB != null) {
					bTextBox.Text = (channelB.ChannelInfo.Sensor != null) ?
						channelB.ChannelInfo.Sensor.GetValueString(channelB.CurrentValue, meterFormat, true) :
						String.Format("U = {0} V", channelB.CurrentValue.ToString(meterFormat));
					                                                                                                           
				} else {
					bTextBox.Text = "NO DATA";
				}
				//
				if (channelC != null) {
					cTextBox.Text = (channelC.ChannelInfo.Sensor != null) ?
						channelC.ChannelInfo.Sensor.GetValueString(channelC.CurrentValue, meterFormat, true) :
						String.Format("U = {0} V", channelC.CurrentValue.ToString(meterFormat));
					                                                                                                           
				} else {
					cTextBox.Text = "NO DATA";
				}
				
				if (!textBoxesInitialised) {
					SetTextBoxes();
					textBoxesInitialised = true;
				}
			}
		}
		
		public void Reset(bool resetDevice)
		{
			if (resetDevice && (device != null)) device.Reset();
			// Reset queues:
			lock (channelA.SyncRoot) { channelA.Clear(); }
			lock (channelB.SyncRoot) { channelB.Clear(); }
			lock (channelC.SyncRoot) { channelC.Clear(); }
			//
			aDataPlotter.Reset();
			bDataPlotter.Reset();
			cDataPlotter.Reset();
			aDataGridView.Rows.Clear();
			bDataGridView.Rows.Clear();
			cDataGridView.Rows.Clear();
			aTextBox.Text = String.Empty;
			bTextBox.Text = String.Empty;
			cTextBox.Text = String.Empty;
		}
		
		public void SetControls()
		{
			// Generic:
			dt = 1.0 / measurement.SamplingFrequency;
			
			fsNumericControl.SetValue(measurement.SamplingFrequency);
			averagesComboBox.SelectedIndex = measurement.Averages.Code;
			xScaleTitleLabelledTextBox.TextBox.Text = measurement.XScaleTitle;
			xPointsNumericControl.MaxValue = maxPoints;
			xPointsNumericControl.SetValue(measurement.XScalePoints);
			xWidthNumericControl.SetValue(dt * xPointsNumericControl.Value);
			refreshRateNumericControl.SetValue(1000.0 / measurement.RefreshInterval);
			// Channel A:
			channelA.Capacity = measurement.DataBufferSize;
			channelAInfo.Sensor = measurement[0].Sensor;
			aSensorControl.Sensor = measurement[0].Sensor;
			HandleSensorUpdatedEvents(aSensorControl, new EventArgs());
			aActiveCheckBox.Checked = measurement[0].IsActive;
			aAutoCheckBox.Checked = measurement[0].IsAutoscaled;
			if (aAutoCheckBox.Checked) {
				aMinNumericControl.Enabled = false;
				aMinNumericControl.SetValue(Double.NaN, "Auto");
				aMaxNumericControl.Enabled = false;
				aMaxNumericControl.SetValue(Double.NaN, "Auto");
				aDataPlotter.AxisY.Autoscaled = true;
			} else {
				aMinNumericControl.Enabled = true;
				aMinNumericControl.SetValue(measurement[0].YMin);
				aMaxNumericControl.Enabled = true;
				aMaxNumericControl.SetValue(measurement[0].YMax);
				aDataPlotter.AxisY.Autoscaled = false;
				aDataPlotter.AxisY.Minimum = measurement[0].YMin;
				aDataPlotter.AxisY.Maximum = measurement[0].YMax;
			}
			if (measurement[0].LevelCrossingDetector != null) {
				aLevelCrossingControl.Set(measurement[0].LevelCrossingDetector, measurement[0].ObjectLength);
				channelA.LevelCrossingDetector = measurement[0].LevelCrossingDetector;
				channelA.LevelCrossings.Capacity = levelCrossingCapacity;
				channelA.LevelCrossings.Updated += HandleLevelCrossingsUpdatedEvents;
			}
			aTextBox.ForeColor = measurement[0].Colour;
			aColourPickerControl.Colour = measurement[0].Colour;
			aDataPlotter.Curves[0].Line.Color = measurement[0].Colour;
			aDataPlotter.AxisX.Title.Text = measurement.XScaleTitle;
			// Channel B:
			channelB.Capacity = measurement.DataBufferSize;
			channelBInfo.Sensor = measurement[1].Sensor;
			bSensorControl.Sensor = measurement[1].Sensor;
			HandleSensorUpdatedEvents(bSensorControl, new EventArgs());
			bActiveCheckBox.Checked = measurement[1].IsActive;
			bAutoCheckBox.Checked = measurement[1].IsAutoscaled;
			if (bAutoCheckBox.Checked) {
				bMinNumericControl.Enabled = false;
				bMinNumericControl.SetValue(Double.NaN, "Auto");
				bMaxNumericControl.Enabled = false;
				bMaxNumericControl.SetValue(Double.NaN, "Auto");
				bDataPlotter.AxisY.Autoscaled = true;
			} else {
				bMinNumericControl.Enabled = true;
				bMinNumericControl.SetValue(measurement[1].YMin);
				bMaxNumericControl.Enabled = true;
				bMaxNumericControl.SetValue(measurement[1].YMax);
				bDataPlotter.AxisY.Autoscaled = false;
				bDataPlotter.AxisY.Minimum = measurement[1].YMin;
				bDataPlotter.AxisY.Maximum = measurement[1].YMax;
			}
			if (measurement[1].LevelCrossingDetector != null) {
				bLevelCrossingControl.Set(measurement[1].LevelCrossingDetector, measurement[1].ObjectLength);
				channelB.LevelCrossingDetector = measurement[1].LevelCrossingDetector;
				channelB.LevelCrossings.Capacity = levelCrossingCapacity;
				channelB.LevelCrossings.Updated += HandleLevelCrossingsUpdatedEvents;
			}
			bTextBox.ForeColor = measurement[1].Colour;
			bColourPickerControl.Colour = measurement[1].Colour;
			bDataPlotter.Curves[0].Line.Color = measurement[1].Colour;
			bDataPlotter.AxisX.Title.Text = measurement.XScaleTitle;
			// Channel C:
			channelC.Capacity = measurement.DataBufferSize;
			channelCInfo.Sensor = measurement[2].Sensor;
			cSensorControl.Sensor = measurement[2].Sensor;
			HandleSensorUpdatedEvents(cSensorControl, new EventArgs());
			cActiveCheckBox.Checked = measurement[2].IsActive;
			cAutoCheckBox.Checked =  measurement[2].IsAutoscaled;
			if (cAutoCheckBox.Checked) {
				cMinNumericControl.Enabled = false;
				cMinNumericControl.SetValue(Double.NaN, "Auto");
				cMaxNumericControl.Enabled = false;
				cMaxNumericControl.SetValue(Double.NaN, "Auto");
				cDataPlotter.AxisY.Autoscaled = true;
			} else {
				cMinNumericControl.Enabled = true;
				cMinNumericControl.SetValue(measurement[2].YMin);
				cMaxNumericControl.Enabled = true;
				cMaxNumericControl.SetValue(measurement[2].YMax);
				cDataPlotter.AxisY.Autoscaled = false;
				cDataPlotter.AxisY.Minimum = measurement[2].YMin;
				cDataPlotter.AxisY.Maximum = measurement[2].YMax;
			}
			if (measurement[2].LevelCrossingDetector != null) {
				cLevelCrossingControl.Set(measurement[2].LevelCrossingDetector, measurement[2].ObjectLength);
				channelC.LevelCrossingDetector = measurement[2].LevelCrossingDetector;
				channelC.LevelCrossings.Capacity = levelCrossingCapacity;
				channelC.LevelCrossings.Updated += HandleLevelCrossingsUpdatedEvents;
			}
			cTextBox.ForeColor = measurement[2].Colour;
			cColourPickerControl.Colour = measurement[2].Colour;
			cDataPlotter.Curves[0].Line.Color = measurement[2].Colour;
			cDataPlotter.AxisX.Title.Text = measurement.XScaleTitle;
			SetPlotters();
			SetChannelVisibility();
			SetTextBoxes();
			SetTableVisibility();
		}
		
		public void SetChannelVisibility()
		{
			TableLayoutRowStyleCollection rows1 = tableLayoutPanel1.RowStyles;
			TableLayoutRowStyleCollection rows2 = tableLayoutPanel2.RowStyles;
			LevelCrossingControl levelCrossingControl = null;
			DataPlotter dataPlotter = null;
			Channel channel = null;
			System.Windows.Forms.Panel panel = null;
			int validChannelCount = measurement.GetActiveChannelCount();
			float percentage;
			
			
			percentage = (validChannelCount > 0) ? 100.0f / validChannelCount : 0.0f;
			
			if (validChannelCount == 0) {
				tableLayoutPanel1.Visible = false;
				tableLayoutPanel2.Visible = false;
				aLevelCrossingControl.Enabled = false;
				bLevelCrossingControl.Enabled = false;
				cLevelCrossingControl.Enabled = false;
				if (channelA.LevelCrossingDetector != null) channelA.LevelCrossingDetector.IsActive = false;
				if (channelB.LevelCrossingDetector != null) channelB.LevelCrossingDetector.IsActive = false;
				if (channelC.LevelCrossingDetector != null) channelC.LevelCrossingDetector.IsActive = false;
			} else {
				tableLayoutPanel1.Visible = true;
				tableLayoutPanel2.Visible = true;
				for (int i = 0; i < 3; i++) {
					switch (i) {
						case 1:
							channel = channelB;
							levelCrossingControl = bLevelCrossingControl;
							panel = bDataGridPanel;
							dataPlotter = bDataPlotter;
							break;
						case 2:
							channel = channelC;
							levelCrossingControl = cLevelCrossingControl;
							panel = cDataGridPanel;
							dataPlotter = cDataPlotter;
							break;
						case 0:
						default:
							channel = channelA;
							levelCrossingControl = aLevelCrossingControl;
							panel = aDataGridPanel;
							dataPlotter = aDataPlotter;
							break;
					}
					if (measurement[i].IsActive) {
						dataPlotter.Visible = true;
						panel.Visible = true;
						levelCrossingControl.Enabled = true;
						//
						rows1[i].SizeType = SizeType.Percent;
						rows1[i].Height = percentage;
						rows2[i].SizeType = SizeType.Percent;
						rows2[i].Height = percentage;
					} else {
						dataPlotter.Visible = false;
						if (channel.LevelCrossingDetector != null) channel.LevelCrossingDetector.IsActive = false;
						panel.Visible = false;
						levelCrossingControl.Enabled = false;
						//
						rows1[i].SizeType = SizeType.Absolute;
						rows1[i].Height = 0.0F;
						rows2[i].SizeType = SizeType.Absolute;
						rows2[i].Height = 0.0F;
					}
				}	
			}
		}
		
		public void SetFtdi()
		{
			string[] ftdiDevices = Ftdi.GetSerialNumbers();
			
			if (ftdi != null) {
				ftdi.Close();
			}
			
			if ((ftdiDevices != null) && (ftdiDevices.Length > 0) && !String.IsNullOrEmpty(ftdiDevices[0])) {
				toolStripStatusLabel1.Text = ftdiDevices[0];
				ftdi = new Ftdi(ftdiDevices[0]);
				ftdi.Open();	
				ftdi.BaudRate = Edaq530.BaudRate;
				ftdi.Handshake = Edaq530.Handshake;
				ftdi.SetLatency((byte)0);
				ftdi.Initialise();
			} else {
				toolStripStatusLabel1.Text = "No Edaq530 device found.";
				MessageBox.Show(
					"No Edaq530 device has been found. Please connect the device, wait about 10 seconds, then select the \"Rescan devices\" option from" +
					" the \"Connexion\" menu or press CTRL + R.", "No Edaq530 device found");
				ftdi = null;
			}
		}
		
		public void SetPlotters()
		{
			aDataPlotter.Interval = measurement.RefreshInterval;
			aDataPlotter.HistoryLength = measurement.XScalePoints;
			aDataPlotter.Curves[0].Line.Color = measurement[0].Colour;
			//
			bDataPlotter.Interval = measurement.RefreshInterval;
			bDataPlotter.HistoryLength = measurement.XScalePoints;
			bDataPlotter.Curves[0].Line.Color = measurement[1].Colour;
			//
			cDataPlotter.Interval = measurement.RefreshInterval;
			cDataPlotter.HistoryLength = measurement.XScalePoints;
			cDataPlotter.Curves[0].Line.Color = measurement[2].Colour;
		}
		
		public void SetTableVisibility()
		{
			bool isVisible = aLevelCrossingControl.IsActive || bLevelCrossingControl.IsActive || cLevelCrossingControl.IsActive;
			float columnWidth = 245.0f;
			TableLayoutColumnStyleCollection columns = tableLayoutPanel1.ColumnStyles;
			
			columns[0].SizeType = SizeType.Percent;
			columns[1].SizeType = SizeType.Absolute;
			
			columns[1].Width = isVisible ? columnWidth : 0.0f;
			columns[0].Width = 100.0f;
		}
		
		public void SetTextBoxes()
		{
			tableLayoutPanel2.Enabled = false;
			//
			int rowCount = measurement.GetActiveChannelCount();
			int rowHeight;
			int rowWidth;
			int margin;
			int textBoxHeight;
			int oldMargin = aTextBox.Margin.Top;
			double heightFactor = Double.MaxValue;
			double widthFactor = Double.MaxValue;
			float resizeFactor;
			double temporary;
			Point location;
			Size size;
			Font oldFont;
			
			if (rowCount == 0) return;
			
			rowHeight = tableLayoutPanel2.Height / rowCount;
			rowWidth = tableLayoutPanel2.Width;
			
			if (measurement[0].IsActive) {
				// Height:
				temporary = (double)(rowHeight - 2 * oldMargin)/aTextBox.PreferredHeight;
				if (temporary < heightFactor) heightFactor = temporary;
				// Width:
				temporary = (double)(rowWidth - 2 * oldMargin)/aTextBox.PreferredSize.Width;
				if (temporary < widthFactor) widthFactor = temporary;
			}
			
			if (measurement[1].IsActive) {
				// Height:
				temporary = (double)(rowHeight - 2 * oldMargin)/bTextBox.PreferredHeight;
				if (temporary < heightFactor) heightFactor = temporary;
				// Width:
				temporary = (double)(rowWidth - 2 * oldMargin)/bTextBox.PreferredSize.Width;
				if (temporary < widthFactor) widthFactor = temporary;
			}
			
			if (measurement[2].IsActive) {
				// Height:
				temporary = (double)(rowHeight - 2 * oldMargin)/cTextBox.PreferredHeight;
				if (temporary < heightFactor) heightFactor = temporary;
				// Width:
				temporary = (double)(rowWidth - 2 * oldMargin)/cTextBox.PreferredSize.Width;
				if (temporary < widthFactor) widthFactor = temporary;
			}
			
			resizeFactor = (float)Math.Min(heightFactor, widthFactor);
			
			oldFont = aTextBox.Font;
			
			aTextBox.Font = new Font(oldFont.FontFamily, oldFont.Size * resizeFactor);
			bTextBox.Font = new Font(oldFont.FontFamily, oldFont.Size * resizeFactor);
			cTextBox.Font = new Font(oldFont.FontFamily, oldFont.Size * resizeFactor);
			
			
			textBoxHeight = aTextBox.Height;
			margin = (rowHeight - textBoxHeight) / 2;
			if (margin < 3) margin = 3;
			
			if (measurement[0].IsActive) {
				aTextBox.Visible = true;
				location = aTextBox.Location;
				size = aTextBox.Size;
				aTextBox.SetBounds(location.X, margin, size.Width, size.Height);
				margin += rowHeight;
			} else {
				aTextBox.Visible = false;
			}
			
			if (measurement[1].IsActive) {
				bTextBox.Visible = true;
				location = bTextBox.Location;
				size = bTextBox.Size;
				bTextBox.SetBounds(location.X, margin, size.Width, size.Height);
				margin += rowHeight;
			} else {
				bTextBox.Visible = false;
			}
			
			if (measurement[2].IsActive) {
				cTextBox.Visible = true;
				location = cTextBox.Location;
				size = cTextBox.Size;
				cTextBox.SetBounds(location.X, margin, size.Width, size.Height);
				margin += rowHeight;
			} else {
				cTextBox.Visible = false;
			}
			tableLayoutPanel2.Enabled = true;
		}
			
		public void Start()
		{
			// Initialise sampling parameters:
			device.SetSamplingFrequency(measurement.SamplingFrequency, measurement.Averages);
			GetSamplingParameters();
			SetPlotters();
			Reset(true);
			//
			startToolStripButton.Text = "Stop";
			//
			device.StartSampling();
			meterRefreshTimer.Start();
			//
			// Disable controls:
			aSensorControl.ReadOnly = true;
			aSensorEditButton.Enabled = false;
			bSensorControl.ReadOnly = true;
			bSensorEditButton.Enabled = false;
			cSensorControl.ReadOnly = true;
			cSensorEditButton.Enabled = false;
			rescanFTDIDevicesToolStripMenuItem.Enabled = false;
			loadMeasurementSetupFromFileToolStripMenuItem.Enabled = false;
			fsNumericControl.Enabled = false;
			averagesComboBox.Enabled = false;
			copyDataToClipboardToolStripMenuItem.Enabled = false;
			saveMeasurementDataToolStripMenuItem.Enabled = false;
			loadSensorFromFileToolStripMenuItem.Enabled = false;
			toChannelAToolStripMenuItem.Enabled = false;
			toChannelBToolStripMenuItem.Enabled = false;
			toChannelCToolStripMenuItem.Enabled = false;
			//
			aDataPlotter.Start();
			bDataPlotter.Start();
			cDataPlotter.Start();
			//
			isSampling = true;
			toolStripStatusLabel1.Text = "Sampling started.";
		}
		
		public void Stop()
		{
			startToolStripButton.Text = "Start";
			startToolStripButton.Enabled = false;
			toolStripStatusLabel1.Text = "Stopping...";
			this.Refresh();
			//
			meterRefreshTimer.Stop();
			aDataPlotter.Stop();
			bDataPlotter.Stop();
			cDataPlotter.Stop();
			//
			if (device != null) {
				device.StopSampling();
				device.SamplingStopped.WaitOne(10000);
			}
			//
			isSampling = false;
			toolStripStatusLabel1.Text = "Sampling stopped.";
			// Enable controls:
			aSensorControl.ReadOnly = false;
			aSensorEditButton.Enabled = true;
			bSensorControl.ReadOnly = false;
			bSensorEditButton.Enabled = true;
			cSensorControl.ReadOnly = false;
			cSensorEditButton.Enabled = true;
			rescanFTDIDevicesToolStripMenuItem.Enabled = true;
			loadMeasurementSetupFromFileToolStripMenuItem.Enabled = true;
			fsNumericControl.Enabled = true;
			averagesComboBox.Enabled = true;
			copyDataToClipboardToolStripMenuItem.Enabled = true;
			saveMeasurementDataToolStripMenuItem.Enabled = true;
			loadSensorFromFileToolStripMenuItem.Enabled = true;
			toChannelAToolStripMenuItem.Enabled = true;
			toChannelBToolStripMenuItem.Enabled = true;
			toChannelCToolStripMenuItem.Enabled = true;
			//
			startToolStripButton.Enabled = true;
		}
		
		public void UpdateDataGridView(DataGridView dataGridView)
		{
			if (dataGridView.InvokeRequired) {
				MethodInvoker method = delegate { UpdateDataGridView(dataGridView); };
				dataGridView.Invoke(method);
			} else {
				Channel channel = null;
				DataGridViewRow row;
				LevelCrossingCycle[] levelCrossings;
				double length = 1.0;
				
				if (dataGridView == aDataGridView) {
					channel = channelA;
					length = aLevelCrossingControl.Length;
				} else if (dataGridView == bDataGridView) {
					channel = channelB;
					length = bLevelCrossingControl.Length;
				} else if (dataGridView == cDataGridView) {
					channel = channelC;
					length = cLevelCrossingControl.Length;
				}
				
				if (channel.LevelCrossingDetector == null) return;
				if (channel.LevelCrossings == null) return;
				
				lock (channel.LevelCrossings.SyncRoot) {
					levelCrossings = channel.LevelCrossings.GetLastPoints(gridViewRows);
				}
				
				if (levelCrossings == null) return;
				if (levelCrossings.Length < 1) return;
				
				((ISupportInitialize)dataGridView).BeginInit();
				
				if (dataGridView.RowCount < levelCrossings.Length) {
					dataGridView.Rows.Add(levelCrossings.Length - dataGridView.RowCount);
				}

				for (int i = 0; i < dataGridView.RowCount && i < levelCrossings.Length; i++) {
					row = dataGridView.Rows[i];
					if (levelCrossings[i] == null) continue;
					row.Cells[0].Value = levelCrossings[i].Start.X;
					if (levelCrossings[i].Next != null) {
						row.Cells[1].Value = levelCrossings[i].Period;
					}
					if (levelCrossings[i].End != null) {
						row.Cells[2].Value = length / levelCrossings[i].PulseWidth;
					}
				}
				((ISupportInitialize)dataGridView).EndInit();
				
				dataGridView.Refresh();
			}
		}
		#endregion
		
		#region EVENT HANDLERS
		protected void HandleButtonClickEvents(object sender, EventArgs e)
		{
			if (sender == startToolStripButton) {
				if (device == null) return;
				//
				if (isSampling) {
					Stop();
				} else {
					Start();
				}
			} else if (sender == aSensorEditButton) {
				using (SensorEditor editor = new SensorEditor(SensorEditorMode.Edit, (channelAInfo.Sensor != null) ? channelAInfo.Sensor.Clone() : null, device, 0)) {
					if (editor.ShowDialog() == DialogResult.OK) {
						channelAInfo.Sensor = editor.Sensor;
						aSensorControl.Sensor = channelAInfo.Sensor;
						measurement[0].Sensor = channelAInfo.Sensor;
						HandleSensorUpdatedEvents(aSensorControl, new EventArgs());
					}
				}
			} else if (sender == bSensorEditButton) {
				using (SensorEditor editor = new SensorEditor(SensorEditorMode.Edit, (channelBInfo.Sensor != null) ? channelBInfo.Sensor.Clone() : null, device, 1)) {
					if (editor.ShowDialog() == DialogResult.OK) {
						channelBInfo.Sensor = editor.Sensor;
						bSensorControl.Sensor = channelBInfo.Sensor;
						measurement[1].Sensor = channelBInfo.Sensor;
						HandleSensorUpdatedEvents(bSensorControl, new EventArgs());
					}
				}
			} else if (sender == cSensorEditButton) {
				using (SensorEditor editor = new SensorEditor(SensorEditorMode.Edit, (channelCInfo.Sensor != null) ? channelCInfo.Sensor.Clone() : null, device, 2)) {
					if (editor.ShowDialog() == DialogResult.OK) {
						channelCInfo.Sensor = editor.Sensor;
						cSensorControl.Sensor = channelCInfo.Sensor;
						measurement[2].Sensor = channelCInfo.Sensor;
						HandleSensorUpdatedEvents(cSensorControl, new EventArgs());
					}
				}
			} else if (sender == resetToolStripButton) {
				Reset(!isSampling);
			} else if (sender == showPanelToolStripButton) {
				if (splitContainer1.Panel2Collapsed == true) {
					splitContainer1.Panel2Collapsed = false;
					showPanelToolStripButton.Text = "Hide side panel";
				} else {
					splitContainer1.Panel2Collapsed = true;
					showPanelToolStripButton.Text = "Show side panel";
				}
				this.Refresh();
			} 
		}
		
		protected void HandleCheckBoxClickEvents(object sender, EventArgs e)
		{
			if (sender == aActiveCheckBox) {
				measurement[0].IsActive = aActiveCheckBox.Checked;
				SetChannelVisibility();
				SetTableVisibility();
				SetTextBoxes();
			} else if (sender == bActiveCheckBox) {
				measurement[1].IsActive = bActiveCheckBox.Checked;
				SetChannelVisibility();
				SetTableVisibility();
				SetTextBoxes();
			} else if (sender == cActiveCheckBox) {
				measurement[2].IsActive = cActiveCheckBox.Checked;
				SetChannelVisibility();
				SetTableVisibility();
				SetTextBoxes();
			} else if (sender == aAutoCheckBox) {
				if (aAutoCheckBox.Checked) {
					measurement[0].IsAutoscaled = true;
					aDataPlotter.AxisY.Autoscaled = true;
					// Min:
					aMinNumericControl.Enabled = false;
					aMinNumericControl.SetValue(Double.NaN, "Auto");
					// Max:
					aMaxNumericControl.Enabled = false;
					aMaxNumericControl.SetValue(Double.NaN, "Auto");
				} else {
					double yMin = aDataPlotter.AxisY.Minimum;
					double yMax = aDataPlotter.AxisY.Maximum;
					//
					measurement[0].IsAutoscaled = false;
					measurement[0].YMin = yMin;
					measurement[0].YMax = yMax;
					//
					aDataPlotter.AxisY.Autoscaled = false;
					// Min:
					aMinNumericControl.Enabled = true;
					aMinNumericControl.SetValue(yMin);
					aDataPlotter.AxisY.Minimum = yMin;
					// Max:
					aMaxNumericControl.Enabled = true;
					aMaxNumericControl.SetValue(yMax);
					aDataPlotter.AxisY.Maximum = yMax;
				}
			} else if (sender == bAutoCheckBox) {
				if (bAutoCheckBox.Checked) {
					measurement[1].IsAutoscaled = true;
					bDataPlotter.AxisY.Autoscaled = true;
					// Min:
					bMinNumericControl.Enabled = false;
					bMinNumericControl.SetValue(Double.NaN, "Auto");
					// Max:
					bMaxNumericControl.Enabled = false;
					bMaxNumericControl.SetValue(Double.NaN, "Auto");
				} else {
					double yMin = bDataPlotter.AxisY.Minimum;
					double yMax = bDataPlotter.AxisY.Maximum;
					//
					measurement[1].IsAutoscaled = false;
					measurement[1].YMin = yMin;
					measurement[1].YMax = yMax;
					//
					bDataPlotter.AxisY.Autoscaled = false;
					// Min:
					bMinNumericControl.Enabled = true;
					bMinNumericControl.SetValue(yMin);
					bDataPlotter.AxisY.Minimum = yMin;
					// Max:
					bMaxNumericControl.Enabled = true;
					bMaxNumericControl.SetValue(yMax);
					bDataPlotter.AxisY.Maximum = yMax;
				}
			} else if (sender == cAutoCheckBox) {
				if (cAutoCheckBox.Checked) {
					measurement[2].IsAutoscaled = true;
					cDataPlotter.AxisY.Autoscaled = true;
					// Min:
					cMinNumericControl.Enabled = false;
					cMinNumericControl.SetValue(Double.NaN, "Auto");
					// Max:
					cMaxNumericControl.Enabled = false;
					cMaxNumericControl.SetValue(Double.NaN, "Auto");
				} else {
					double yMin = cDataPlotter.AxisY.Minimum;
					double yMax = cDataPlotter.AxisY.Maximum;
					//
					measurement[2].IsAutoscaled = false;
					measurement[2].YMin = yMin;
					measurement[2].YMax = yMax;
					//
					cDataPlotter.AxisY.Autoscaled = false;
					// Min:
					cMinNumericControl.Enabled = true;
					cMinNumericControl.SetValue(yMin);
					cDataPlotter.AxisY.Minimum = yMin;
					// Max:
					cMaxNumericControl.Enabled = true;
					cMaxNumericControl.SetValue(yMax);
					cDataPlotter.AxisY.Maximum = yMax;
				}
			}
		}
		
		protected void HandleColourChangedEvents(object sender, EventArgs e)
		{
			if (sender == aColourPickerControl) {
				measurement[0].Colour = aColourPickerControl.Colour;
				aTextBox.ForeColor = measurement[0].Colour;
				aDataPlotter.Curves[0].Line.Color = measurement[0].Colour;
			} else if (sender == bColourPickerControl) {
				measurement[1].Colour = bColourPickerControl.Colour;
				bTextBox.ForeColor = measurement[1].Colour;
				bDataPlotter.Curves[0].Line.Color = measurement[1].Colour;
			} else if (sender == cColourPickerControl) {
				measurement[2].Colour = cColourPickerControl.Colour;
				cTextBox.ForeColor = measurement[2].Colour;
				cDataPlotter.Curves[0].Line.Color = measurement[2].Colour;
			}
		}
		
		protected void HandleContextMenuItemClickEvents(object sender, EventArgs e)
		{
			if (sender == copyContentToClipboardToolStripMenuItem) {
				DataGridView dataGridview = tableContextMenuStrip.SourceControl as DataGridView;
				
				CopyToClipBoard(dataGridview);
			} else if (sender == clearToolStripMenuItem) {
				DataGridView dataGridView = tableContextMenuStrip.SourceControl as DataGridView;
				if (dataGridView == aDataGridView) {
					if (channelA.LevelCrossingDetector != null) {
						lock (channelA.LevelCrossings.SyncRoot) {
							channelA.LevelCrossings.Clear();
						}
					}
				} else if (dataGridView == bDataGridView) {
					if (channelB.LevelCrossingDetector != null) {
						lock (channelB.LevelCrossings.SyncRoot) {
							channelB.LevelCrossings.Clear();
						}
					}
				} else if (dataGridView == cDataGridView) {
					if (channelC.LevelCrossingDetector != null) {
						lock (channelC.LevelCrossings.SyncRoot) {
							channelC.LevelCrossings.Clear();
						}
					}
				}
				
				dataGridView.Rows.Clear();
			} else if (sender == saveAsImageToolStripMenuItem) {
				DataPlotter plotter = plotterContextMenuStrip.SourceControl as DataPlotter;
				
				using (SaveFileDialog saveDialogue = new SaveFileDialog()) {
					saveDialogue.Filter = "Portable network graphics format (*.png)|*.png| Jpeg format (*.jpg)|*.jpg";
					if (saveDialogue.ShowDialog() == DialogResult.OK) {
						ImageFormat format;
						switch (saveDialogue.FilterIndex) {
							case 2:
								format = ImageFormat.Jpeg;
								break;
							case 1:
							default:
								format = ImageFormat.Png;
								break;
						}
						plotter.SaveImage(saveDialogue.FileName, format);
					}
				}
			} else if (sender == copyDataToClipboardToolStripMenuItem) {
				DataPlotter plotter = plotterContextMenuStrip.SourceControl as DataPlotter;
				
				plotter.CopyToClipBoard();
			} else if (sender == resetToolStripMenuItem) {
				Reset(!isSampling);
			}
		}
		
		protected void HandleLevelCrossingsUpdatedEvents(object sender, EventArgs e)
		{
			if (sender == channelA.LevelCrossings) {
				UpdateDataGridView(aDataGridView);
			} else if (sender == channelB.LevelCrossings) {
				UpdateDataGridView(bDataGridView);
			} else if (sender == channelC.LevelCrossings) {
				UpdateDataGridView(cDataGridView);
			}
		}
		
		protected void HandleLevelCrossingControlUpdatedEvents(object sender, EventArgs e)
		{
			if (sender == aLevelCrossingControl) {
				if (channelA.LevelCrossingDetector == null) {
					channelA.LevelCrossingDetector = aLevelCrossingControl.CreateLevelCrossingDetector();
				} else {
					aLevelCrossingControl.SetLevelCrossingDetector(channelA.LevelCrossingDetector);
				}
				measurement[0].LevelCrossingDetector = channelA.LevelCrossingDetector;
				measurement[0].ObjectLength = aLevelCrossingControl.Length;
			} else if (sender == bLevelCrossingControl) {
				if (channelB.LevelCrossingDetector == null) {
					channelB.LevelCrossingDetector = bLevelCrossingControl.CreateLevelCrossingDetector();
				} else {
					bLevelCrossingControl.SetLevelCrossingDetector(channelB.LevelCrossingDetector);
				}
				measurement[1].LevelCrossingDetector = channelB.LevelCrossingDetector;
				measurement[1].ObjectLength = bLevelCrossingControl.Length;
			} else if (sender == cLevelCrossingControl) {
				if (channelC.LevelCrossingDetector == null) {
					channelC.LevelCrossingDetector = cLevelCrossingControl.CreateLevelCrossingDetector();
				} else {
					cLevelCrossingControl.SetLevelCrossingDetector(channelC.LevelCrossingDetector);
				}
				measurement[2].LevelCrossingDetector = channelC.LevelCrossingDetector;
				measurement[2].ObjectLength = cLevelCrossingControl.Length;
			} else {
				return;
			}
			
			SetTableVisibility();
		}
		
		protected void HandleMenuItemClickEvents(object sender, EventArgs e)
		{
			if (sender == rescanFTDIDevicesToolStripMenuItem) {
				SetFtdi();
				InitialiseDevice();
			}  else if (sender == fromChannelAToolStripMenuItem) {
				using (SaveFileDialog dialogue = new SaveFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					if (channelAInfo.Sensor != null) dialogue.FileName = channelAInfo.Sensor.Name;
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						if (channelAInfo.Sensor != null) {
							channelAInfo.Sensor.Save(dialogue.FileName);
						}
					}
				}
			} else if (sender == fromChannelBToolStripMenuItem) {
				using (SaveFileDialog dialogue = new SaveFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					if (channelBInfo.Sensor != null) dialogue.FileName = channelBInfo.Sensor.Name;
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						if (channelBInfo.Sensor != null) {
							channelBInfo.Sensor.Save(dialogue.FileName);
						}
					}
				}
			} else if (sender == fromChannelCToolStripMenuItem) {
				using (SaveFileDialog dialogue = new SaveFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					if (channelCInfo.Sensor != null) dialogue.FileName = channelCInfo.Sensor.Name;
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						if (channelCInfo.Sensor != null) {
							channelCInfo.Sensor.Save(dialogue.FileName);
						}
					}
				}
			} else if (sender == toChannelAToolStripMenuItem) {
				using (OpenFileDialog dialogue = new OpenFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						channelAInfo.Sensor = Sensor.Open(dialogue.FileName);
						aSensorControl.Sensor = channelAInfo.Sensor;
						measurement[0].Sensor = channelAInfo.Sensor;
					}
				}
			} else if (sender == toChannelBToolStripMenuItem) {
				using (OpenFileDialog dialogue = new OpenFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						channelBInfo.Sensor = Sensor.Open(dialogue.FileName);
						bSensorControl.Sensor = channelBInfo.Sensor;
						measurement[1].Sensor = channelBInfo.Sensor;
					}
				}
			} else if (sender == toChannelCToolStripMenuItem) {
				using (OpenFileDialog dialogue = new OpenFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						channelCInfo.Sensor = Sensor.Open(dialogue.FileName);
						cSensorControl.Sensor = channelCInfo.Sensor;
						measurement[2].Sensor = channelCInfo.Sensor;
					}
				}
			} else if (sender == saveMeasurementSetupToolStripMenuItem) {
				using (SaveFileDialog dialogue = new SaveFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						if (measurement != null) measurement.Save(dialogue.FileName);
					}
				}
			} else if (sender == loadMeasurementSetupFromFileToolStripMenuItem) {
				using (OpenFileDialog dialogue = new OpenFileDialog()) {
					dialogue.Filter = "Xml files (*.xml)|*.xml";
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						LoadMeasurement(dialogue.FileName);
					}
				}
			} else if (sender == saveMeasurementDataToolStripMenuItem) {
				using (SaveFileDialog dialogue = new SaveFileDialog()) {
					dialogue.SupportMultiDottedExtensions = true;
					dialogue.Filter = "Text files (*.data.txt)|*.data.txt";
					
					if (dialogue.ShowDialog() == DialogResult.OK) {
						if (channels != null) {
							channels.Save(dialogue.FileName,"f4", "\t");
						}
					}
				}
			}
		}
		
		protected void HandleValueChangedEvents(object sender, EventArgs e)
		{
			if (sender == fsNumericControl) {
				if (isSampling) return;
				if (!device.SamplingStopped.WaitOne(0)) return;
				if (!Double.IsNaN(fsNumericControl.Value)) {
					device.SetSamplingFrequency(fsNumericControl.Value, measurement.Averages);
					GetSamplingParameters();
					SetPlotters();
				}
			} else if (sender == aMinNumericControl) {	
				if (Double.IsNaN(aDataPlotter.AxisY.Maximum) ||
					    (aMinNumericControl.Value < aDataPlotter.AxisY.Maximum)) {
					aDataPlotter.AxisY.Minimum = aMinNumericControl.Value;
					measurement[0].YMin = aMinNumericControl.Value;
				} else {
					aMinNumericControl.SetValue(aDataPlotter.AxisY.Minimum, "Auto");
				}
			} else if (sender == aMaxNumericControl) {
				if (aMaxNumericControl.Value > aDataPlotter.AxisY.Minimum) {
					aDataPlotter.AxisY.Maximum = aMaxNumericControl.Value;
					measurement[0].YMax = aMaxNumericControl.Value;
				} else {
					aMaxNumericControl.SetValue(aDataPlotter.AxisY.Maximum, "Auto");
				}
			} else if (sender == bMinNumericControl) {	
				if (Double.IsNaN(bDataPlotter.AxisY.Maximum) ||
					    (bMinNumericControl.Value < bDataPlotter.AxisY.Maximum)) {
					bDataPlotter.AxisY.Minimum = bMinNumericControl.Value;
					measurement[1].YMin = bMinNumericControl.Value;
				} else {
					bMinNumericControl.SetValue(bDataPlotter.AxisY.Minimum, "Auto");
				}
			} else if (sender == bMaxNumericControl) {
				if (bMaxNumericControl.Value > bDataPlotter.AxisY.Minimum) {
					bDataPlotter.AxisY.Maximum = bMaxNumericControl.Value;
					measurement[1].YMax = bMaxNumericControl.Value;
				} else {
					bMaxNumericControl.SetValue(bDataPlotter.AxisY.Maximum, "Auto");
				}
			} else if (sender == cMinNumericControl) {	
				if (Double.IsNaN(cDataPlotter.AxisY.Maximum) ||
					    (cMinNumericControl.Value < cDataPlotter.AxisY.Maximum)) {
					cDataPlotter.AxisY.Minimum = cMinNumericControl.Value;
					measurement[2].YMin = cMinNumericControl.Value;
				} else {
					cMinNumericControl.SetValue(cDataPlotter.AxisY.Minimum, "Auto");
				}
			} else if (sender == cMaxNumericControl) {
				if (cMaxNumericControl.Value > cDataPlotter.AxisY.Minimum) {
					cDataPlotter.AxisY.Maximum = cMaxNumericControl.Value;
					measurement[2].YMax = cMaxNumericControl.Value;
				} else {
					cMaxNumericControl.SetValue(cDataPlotter.AxisY.Maximum, "Auto");
				}
			} else if (sender == xScaleTitleLabelledTextBox) {
				aDataPlotter.AxisX.Title.Text = xScaleTitleLabelledTextBox.TextBox.Text;
				bDataPlotter.AxisX.Title.Text = xScaleTitleLabelledTextBox.TextBox.Text;
				cDataPlotter.AxisX.Title.Text = xScaleTitleLabelledTextBox.TextBox.Text;
				measurement.XScaleTitle = xScaleTitleLabelledTextBox.TextBox.Text;
			} else if (sender == xWidthNumericControl) {
				if (xWidthNumericControl.Value == 0.0) {
					xWidthNumericControl.TextBox.Text = String.Empty;
					return;
				}
				double dt = 1.0 / measurement.SamplingFrequency;
				int points = (int)Math.Round(xWidthNumericControl.Value / dt, MidpointRounding.AwayFromZero);
				if (points > maxPoints) points = maxPoints;
				xPointsNumericControl.SetValue(points);
				xWidthNumericControl.SetValue(points * dt);
				measurement.XScalePoints = points;
				SetPlotters();
			} else if (sender == xPointsNumericControl) {
				if (xPointsNumericControl.Value < 2.0) {
					xPointsNumericControl.TextBox.Text = String.Empty;
					return;
				}
				double dt = 1.0 / measurement.SamplingFrequency;
				int points = (int)xPointsNumericControl.Value;
				xWidthNumericControl.SetValue(points * dt);
				measurement.XScalePoints = points;
				SetPlotters();
			} else if (sender == refreshRateNumericControl) {
				measurement.RefreshInterval = (int)Math.Round(1000.0 / refreshRateNumericControl.Value, MidpointRounding.AwayFromZero);
				aDataPlotter.Interval = measurement.RefreshInterval;
				bDataPlotter.Interval = measurement.RefreshInterval;
				cDataPlotter.Interval = measurement.RefreshInterval;
			}
		}
		
		protected void HandlePanelExpandingEvents(object sender, XPanderStateChangeEventArgs e)
		{
			xPanderPanelList1.Refresh();
			
			if (sender is XPanderPanel) {
				XPanderPanel panel = sender as XPanderPanel;
				panel.Refresh();
			}
		}
	
		protected void HandleSelectionChangeCommittedEvents(object sender, EventArgs e)
		{
			if (sender == averagesComboBox) {
				device.SetSamplingFrequency(measurement.SamplingFrequency, Averages.FromCode((byte)averagesComboBox.SelectedIndex));
				GetSamplingParameters();
			}
		}
		
		protected void HandleSensorRecreatedEvents(object sender, EventArgs e)
		{
			if (sender == aSensorControl) {
				measurement[0].Sensor = aSensorControl.Sensor;
				channelAInfo.Sensor = aSensorControl.Sensor;
				HandleSensorUpdatedEvents(sender, e);
			} else if (sender == bSensorControl) {
				measurement[1].Sensor = bSensorControl.Sensor;
				channelBInfo.Sensor = bSensorControl.Sensor;
				HandleSensorUpdatedEvents(sender, e);
			} else if (sender == cSensorControl) {
				measurement[2].Sensor = cSensorControl.Sensor;
				channelCInfo.Sensor = cSensorControl.Sensor;
				HandleSensorUpdatedEvents(sender, e);
			}
		}
		
		protected void HandleSensorUpdatedEvents(object sender, EventArgs e)
		{
			if (sender == aSensorControl) {
				if (channelAInfo.Sensor == null) {
					aDataPlotter.AxisY.Title.Text = "U [V]";
				} else {
					aDataPlotter.AxisY.Title.Text = channelAInfo.Sensor.GetLabel();
				}
			} else if (sender == bSensorControl) {
				if (channelBInfo.Sensor == null) {
					bDataPlotter.AxisY.Title.Text = "U [V]";
				} else {
					bDataPlotter.AxisY.Title.Text = channelBInfo.Sensor.GetLabel();
				}
			} else if (sender == cSensorControl) {
				if (channelCInfo.Sensor == null) {
					cDataPlotter.AxisY.Title.Text = "U [V]";
				} else {
					cDataPlotter.AxisY.Title.Text = channelCInfo.Sensor.GetLabel();
				}
			}
		}
		
		protected void HandleSplitterMovedEvents(object sender, SplitterEventArgs e)
		{
			this.Refresh();
		}
		#endregion
		
		#region OVERRIDES
		[System.ComponentModel.EditorBrowsableAttribute()]
		protected override void OnResize(EventArgs e)
		{
			WindowsAPI.SuspendRedraw(this.Handle);
			base.OnResize(e);
			SetTextBoxes();
			WindowsAPI.ResumeRedraw(this.Handle);
			this.Refresh();
		}
		
		[System.ComponentModel.EditorBrowsableAttribute()]
		protected override void OnFormClosing(FormClosingEventArgs e)
		{
			if (isSampling) Stop();
			if ((ftdi != null) && ftdi.IsOpen) ftdi.Close();
			measurement.Save("LastSettings.xml");
			base.OnFormClosing(e);
		}
		#endregion
	}
}
