﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2010.02.03.
 * Idő: 10:34
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections.Generic;
using System.Xml;
//
using Noise.DataAcquisition;
using TetheredSun.Core;

namespace GenericMonitor
{
	/// <summary>
	/// Description of Measurement.
	/// </summary>
	public class Measurement : IXmlConvertible
	{
		protected List<ChannelSettings> channels;
		protected string xScaleTitle = "t [s]";
		protected int xScalePoints = 100;
		protected double samplingFrequency = 100.0;
		protected Averages averages = Averages.One;
		protected int dataBufferSize = 100000;
		protected int refreshInterval = 100;
		
		
		public Measurement()
		{
			channels = new List<ChannelSettings>(3);
			
			channels.Add(new ChannelSettings("A", true, null));
			channels.Add(new ChannelSettings("B", true, null));
			channels.Add(new ChannelSettings("C", true, null));
		}
		
		
		public Averages Averages {
			get { return averages; }
			set { averages = value; }
		}
		
		public ChannelSettings this[int index] {
			get { return channels[index]; }
			set { channels[index] = value; }
		}
		
		
		public int DataBufferSize {
			get { return dataBufferSize; }
			set { dataBufferSize = value; }
		}
		
		
		public int RefreshInterval {
			get { return refreshInterval; }
			set { refreshInterval = value; }
		}
		
		
		public double SamplingFrequency {
			get { return samplingFrequency; }
			set { samplingFrequency = value; }
		}
		
		public string XScaleTitle {
			get { return xScaleTitle; }
			set { xScaleTitle = value; }
		}
		
		public int XScalePoints {
			get { return xScalePoints; }
			set { xScalePoints = value; }
		}
				

		public void FromXml(XmlElement xmlElement)
		{
			XmlNodeList channelsXml = xmlElement.SelectNodes("child::ChannelSettings");
			XmlNode child;
			string channelName;
			double parsedDouble;
			int parsedInt;
			ChannelSettings current;
			
			child = xmlElement.SelectSingleNode("child::SamplingFrequency");
			samplingFrequency = Double.TryParse(child.InnerXml, out parsedDouble) ? parsedDouble : 100.0;
			
			child = xmlElement.SelectSingleNode("child::RefreshInterval");
			refreshInterval = Int32.TryParse(child.InnerXml, out parsedInt) ? parsedInt : 100;
			
			child = xmlElement.SelectSingleNode("child::Averages");
			if (child != null) {
				averages = Averages.Undefined;
				averages.FromXml(child as XmlElement);
			} 
			
			child = xmlElement.SelectSingleNode("child::XScale/Title");
			xScaleTitle = child.InnerXml;
			
			child = xmlElement.SelectSingleNode("child::XScale/Points");
			xScalePoints = Int32.TryParse(child.InnerXml, out parsedInt) ? parsedInt : 100;
			
			child = xmlElement.SelectSingleNode("child::DataBufferSize");
			dataBufferSize = Int32.TryParse(child.InnerXml, out parsedInt) ? parsedInt : RingBuffer<double>.DefaultCapacity;
			

			for (int i = 0; i < channelsXml.Count; i++) {
				child = channelsXml[i].SelectSingleNode("child::Name");
				channelName = child.InnerXml;
				
				current = new ChannelSettings();
				current.FromXml(channelsXml[i] as XmlElement);
				this[i] = current;
			}
		}
		
		public int GetActiveChannelCount()
		{
			int activeChannelCount = 0;
			
			foreach (ChannelSettings channel in channels)
			{
				if (channel.IsActive) activeChannelCount++;
			}
			
			return activeChannelCount;
		}
		
		public static Measurement Open(string path)
		{
			Measurement measurement = new Measurement();
			XmlDocument document = new XmlDocument();
			
			document.Load(path);
			measurement.FromXml(document.DocumentElement);
			
			return measurement;
		}
		
		public void Save(string path)
		{
			XmlDocument document = new XmlDocument();
			
			document.AppendChild(document.CreateXmlDeclaration("1.0", "utf-8", "yes"));
			document.AppendChild(this.ToXml(document));
			
			document.Save(path);
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement measurementXml = context.CreateElement("Measurement");
			XmlElement averagesXml = (averages != null) ? averages.ToXml(context) : null;
			XmlElement samplingFrequencyXml = context.CreateElement("SamplingFrequency");
			XmlElement refreshIntervalXml = context.CreateElement("RefreshInterval");
			XmlElement xScaleXml = context.CreateElement("XScale");
			XmlElement titleXml = context.CreateElement("Title");
			XmlElement pointsXml = context.CreateElement("Points");
			XmlElement dataBufferSizeXml = context.CreateElement("DataBufferSize");
			
			samplingFrequencyXml.SetAttribute("unit", "Hz");
			samplingFrequencyXml.InnerXml = samplingFrequency.ToString();
			measurementXml.AppendChild(samplingFrequencyXml);
			
			refreshIntervalXml.SetAttribute("unit", "ms");
			refreshIntervalXml.InnerXml = refreshInterval.ToString();
			measurementXml.AppendChild(refreshIntervalXml);
			
			if (averagesXml != null) measurementXml.AppendChild(averagesXml);
			
			titleXml.InnerXml = xScaleTitle;
			xScaleXml.AppendChild(titleXml);
			pointsXml.InnerXml = xScalePoints.ToString();
			xScaleXml.AppendChild(pointsXml);
			measurementXml.AppendChild(xScaleXml);
			
			dataBufferSizeXml.InnerXml = dataBufferSize.ToString();
			measurementXml.AppendChild(dataBufferSizeXml);
			
			
			foreach (ChannelSettings channelSettings in channels) {
				if (channelSettings != null) {
					measurementXml.AppendChild(channelSettings.ToXml(context));
				}
			}
			
			return measurementXml;
		}
	}
}
