/*
 * Created by SharpDevelop.
 * User: phil
 * Date: 2010.06.17.
 * Time: 11:00
 * 
 * To change this template use Tools | Options | Coding | Edit Standard Headers.
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Runtime.InteropServices;
//
using TetheredSun.Core;

namespace Noise.Controls
{
	/// <summary>
	/// Description of DataPointList.
	/// </summary>
	public class DataPointList : IList<Point<double>>
	{
		public static int DefaultCapacity = 100;
		protected static double[] emptyArray;
		protected int count;
		protected double[] x;
		protected double[] y;
		protected object syncRoot = new Object();
		
		#region NESTED CLASSES
		[Serializable, StructLayout(LayoutKind.Sequential)]
		public struct Enumerator : IEnumerator<Point<double>>, IDisposable, IEnumerator
		{
			private DataPointList list;
		    private int index;
		    private Point<double> current;
		    
		    
		    internal Enumerator(DataPointList list)
			{
			    this.list = list;
			    this.index = 0;
			    this.current = null;
			}
			
			public Point<double> Current {
		    	get { return current; }
			}
			
			object IEnumerator.Current {
				get {
					if ((index == 0) || (index == (list.count + 1))) {
						//ThrowHelper.ThrowInvalidOperationException(ExceptionResource.InvalidOperation_EnumOpCantHappen);
						Exception exception = new InvalidOperationException();
						throw exception;
					}
					return current;
				}
			}
			
			public void Dispose()
			{
			}
			
			public bool MoveNext()
			{
				 if (index < list.count) {
				 	current = list[index];
				 	index++;
				 	return true;
				 }
				 return MoveNextRare();

			}
			
			public void Reset()
			{
				index = 0;
				current = null;
			}
			
			private bool MoveNextRare()
			{
				index = list.count + 1;
				current = null;
				return false;
			} 
		}
		#endregion
		
		static DataPointList()
		{
			DataPointList.emptyArray = new double[0];
		}
		
		public DataPointList(int capacity)
		{
			count = 0;
			x = new double[capacity];
			y = new double[capacity];
		}
		
		
		public int Capacity {
		    get { return y.Length;}
		    set {
		        if (value != y.Length) {
		    		if (value > 0) {
		    			int startIndex = (value < count) ?  count - value : 0;	// If 'value' is less than 'count', the last 'value' number of points are to be kept.
		    			int copyCount = Math.Min(count, value);
		                double[] destinationX = new double[value];
		                double[] destinationY = new double[value];
		                if (this.count > 0) {
		                    Array.Copy(x, startIndex, destinationX, 0, copyCount);
		                    Array.Copy(y, startIndex, destinationY, 0, copyCount);
		                }
		                x = destinationX;
		                y = destinationY;
		                count = copyCount;	// TODO: Not done like this in List<T>.
		            } else {
		                x = DataPointList.emptyArray;
		                y = DataPointList.emptyArray;
		                count = 0;			// TODO: Not done like this in List<T>.
		            }
		        }
		    }
		}
 
		public object SyncRoot {
			get { return syncRoot; }
		}
		

		public void Add(double x, double y)
		{
			if (count == this.y.Length) {
				EnsureCapacity(count + 1);
			}
			
			this.x[count] = x;
			this.y[count] = y;
			count++;
		}
	
		public void SetItems(double[] x, double[] y)
		{
			if ((x == null) || (y == null)) {
				Clear();
				return;
			}
			
			if ((x.Length == 0) || (y.Length == 0)) {
				Clear();
				return;
			}
			
			this.x = x;
			this.y = y;
			
			if (this.x.Length < this.y.Length) {
				this.y = new double[this.x.Length];
				Array.Copy(y, this.y, this.x.Length);
			} else if (this.x.Length > this.y.Length) {
				this.x = new double[this.y.Length];
				Array.Copy(x, this.x, this.y.Length);
			}
			
			count = this.y.Length;
		}
		
		public void SetItems(double[] y, double x0, double dx)
		{
			if ((y == null) || (y.Length == 0)) {
				Clear();
				return;
			}
			
			this.y = y;
			count = this.y.Length;
			x = new double[count];
			
			for (int i = 0; i < count; i++) {
				x[i] = x0 + i * dx;
			}
		}
		
		public bool UpdateBounds(out double xMinimum, out double xMaximum, out double yMinimum, out double yMaximum)
		{
			if (this.count < 1) {
				xMinimum = Double.NaN;
				xMaximum = Double.NaN;
				yMinimum = Double.NaN;
				yMaximum = Double.NaN;
				return false;
			}
			
			xMinimum = x[0];
			xMaximum = x[0];
			yMinimum = y[0];
			yMaximum = y[0];
	
			for (int i = 0; i < this.count; i++) {
				if (x[i] < xMinimum) {
					xMinimum = x[i];
				} else if (x[i] > xMaximum) {
					xMaximum = x[i];
				}
				
				if (y[i] < yMinimum) {
					yMinimum = y[i];
				} else if (y[i] > yMaximum) {
					yMaximum = y[i];
				}
			}
			
			return true;
		}
		
		#region INTERFACE IMPLEMENTATION
		public Point<double> this[int index]
		{
			get {
				if (index >= count) {
					throw new ArgumentOutOfRangeException();
				}
				return new Point<double>(x[index], y[index]);
			}
			set {
				if (index >= count) {
					throw new ArgumentOutOfRangeException();
				}
				x[index] = value.X;
				y[index] = value.Y;
			}
		}
		
		public int Count {
			get { return count; }
		}
		
		public bool IsReadOnly {
			get { return false; }
		}
		
		public int IndexOf(Point<double> item)
		{
			int index = Array.IndexOf(x, item.X, 0, count);
			if (index < 0) {
				return -1;
			} else {
				return (y[index] == item.Y) ? index : -1;
			}
		}
		
		public void Insert(int index, Point<double> item)
		{
			if (index > count) {
				Exception exception = new ArgumentOutOfRangeException("index", index,
				                                                      "Index must be less than or equal to count.");
				throw exception;
			}
			if (count == y.Length) {
				this.EnsureCapacity(count + 1);
			}
			if (index < count) {
				Array.Copy(x, index, x, index + 1, count - index);
				Array.Copy(y, index, y, index + 1, count - index);
			}
			x[index] = item.X;
			y[index] = item.Y;
			count++;
		}
		
		public void RemoveAt(int index)
		{
			if (index >= count) {
				Exception exception = new ArgumentOutOfRangeException("index", index,
				                                                      "Index must be less than or equal to count.");
				throw exception;
			}
			count--;
			if (index < count)
			{
				Array.Copy(x, index + 1, x, index, count - index);
				Array.Copy(y, index + 1, y, index, count - index);
			}
			x[count] = 0.0;
			y[count] = 0.0;
		}
		
		public void Add(Point<double> item)
		{
			Add(item.X, item.Y);
		}
		
		public void Clear()
		{
			 if (count > 0) {
			 	Array.Clear(x, 0, count);
			 	Array.Clear(y, 0, count);
			 	count = 0;
			 }

		}
		
		public bool Contains(Point<double> item)
		{
			if (Object.ReferenceEquals(item, null)) return false;
			for (int i = 0; i < count; i++) {
				if (x[i] == item.X) {
					if (y[i] == item.Y) return true;
				}
			}
			return false;
		}
		
		public void CopyTo(Point<double>[] array, int arrayIndex)
		{
			for (int i = 0; i < count; i++) {
				array[arrayIndex + i] = new Point<double>(x[i], y[i]);
			}
		}
		
		public bool Remove(Point<double> item)
		{
			int index = this.IndexOf(item);
			if (index >= 0) {
				RemoveAt(index);
				return true;
			}
			return false;
		}
		
		public IEnumerator<Point<double>> GetEnumerator()
		{
			return new Enumerator(this);
		}
		
		IEnumerator IEnumerable.GetEnumerator()
		{
			return new Enumerator(this);
		}
		#endregion
		
		private void EnsureCapacity(int min)
		{
		    if (y.Length < min)
		    {
		        int num = (y.Length == 0) ? 4 : (y.Length * 2);
		        if (num < min) {
		            num = min;
		        }
		       Capacity = num;
		    }
		}
	}
}
