﻿/*
 * Created by SharpDevelop.
 * User: phil
 * Date: 2010.03.30.
 * Time: 15:01
 * 
 * To change this template use Tools | Options | Coding | Edit Standard Headers.
 */
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Drawing.Text;
using System.Windows.Forms;
//
using TetheredSun.Core;

namespace Noise.Controls
{
	public class NoiseGraph : Control
	{
		protected Font				scaleFont = new Font("Trebuchet MS", 12.0f);
		protected Pen				axisPen = new Pen(Color.Black, 1.0f);
		protected Pen				tickPen = new Pen(Color.Black, 1.0f);
		protected Pen				gridPen = new Pen(Color.Gray, 1.0f);
		protected Pen				debugPen = new Pen(Color.Red, 2.0f);
		protected Brush				scaleBrush = new SolidBrush(Color.Black);
		//
		protected Axis				axisX = new Axis(AxisType.X);
		protected Axis				axisY = new Axis(AxisType.Y);
		//
		protected Rectangle			drawRectangle;
		protected Rectangle			axisFrameRectangle;
		protected Rectangle 		curveRectangle;
		protected double			xMax;
		protected double			xMin;
		protected double			yMin;
		protected double			yMax;
		protected double			x0;
		protected double			y0;
		protected double			dx;
		protected double			dy;
		protected int				numberOfPoints;
		protected int				scalePadding = 5;
		protected bool				enforceTickBounds = true;
		protected bool				boundsUndefined = true;
		protected CurveList			curves = new CurveList();
		protected SmoothingMode 	lineSmoothingMode = SmoothingMode.Default;
		protected SmoothingMode 	symbolSmoothingMode = SmoothingMode.Default;
		protected TextRenderingHint	textRenderingHint = TextRenderingHint.SystemDefault;
		protected GraphicsState		state;
		
		public NoiseGraph() : base()
		{
			this.Padding = new Padding(20);
			this.DoubleBuffered = true;
			SetStyle(ControlStyles.SupportsTransparentBackColor, true);
			//
			axisX.Title.Text = "X axis";
			axisY.Title.Text = "Y axis";
			//
			gridPen.DashStyle = DashStyle.Dot;
			gridPen.DashCap = DashCap.Round;
		}
		
		public Axis AxisX {
			get { return axisX; }
		}
		
		public Axis AxisY {
			get { return axisY; }
		}
		
		public CurveList Curves {
			get { return curves; }
		}
		
		
		public Font ScaleFont {
			get { return scaleFont; }
			set { scaleFont = value; }
		}
		
		public Pen AxisPen {
			get { return axisPen; }
			set { axisPen = value; }
		}
		
		public Pen TickPen {
			get { return tickPen; }
			set { tickPen = value; }
		}
		
		public Pen GridPen {
			get { return gridPen; }
			set { gridPen = value; }
		}
		
		public Brush ScaleBrush {
			get { return scaleBrush; }
			set { scaleBrush = value; }
		}
		
		public SmoothingMode LineSmoothingMode {
			get { return lineSmoothingMode; }
			set { lineSmoothingMode = value; }
		}
		
		public SmoothingMode SymbolSmoothingMode {
			get { return symbolSmoothingMode; }
			set { symbolSmoothingMode = value; }
		}
		
		public TextRenderingHint TextRenderingHint {
			get { return textRenderingHint; }
			set { textRenderingHint = value; }
		}
		
		
		public void SaveImage(string path, ImageFormat format)
		{
			Bitmap image = new Bitmap(this.ClientRectangle.Width, this.ClientRectangle.Height, PixelFormat.Format32bppArgb);
			
			this.DrawToBitmap(image, this.ClientRectangle);
			image.Save(path, format);
		}
		
		
		protected void UpdateAxisBounds()
		{
			if (curves.Count < 1) {
				boundsUndefined = true;
				return;
			}
			
			if (curves[0].Points.Count < 1) {
				boundsUndefined = true;
				return;
			}
			
			double xMinimum;
			double xMaximum;
			double yMinimum;
			double yMaximum;
			
			boundsUndefined = false;
			lock (curves[0].Points.SyncRoot) {
				curves[0].Points.UpdateBounds(out xMinimum, out xMaximum, out yMinimum, out yMaximum);
			}
			
			if (axisX.Autoscaled) {
				xMin = xMinimum;
				xMax = xMaximum;
				axisX.Minimum = xMin;
				axisX.Maximum = xMax;
			} else {
				xMin = axisX.Minimum;
				xMax = axisX.Maximum;
			}
			
			if (axisY.Autoscaled) {
				yMin = yMinimum;
				yMax = yMaximum;
				axisY.Minimum = yMin;
				axisY.Maximum = yMax;
			} else {
				yMin = axisY.Minimum;
				yMax = axisY.Maximum;
			}
			
			boundsUndefined = Double.IsNaN(xMin) || Double.IsNaN(xMax) || Double.IsNaN(yMin) || Double.IsNaN(yMax);
		}
		
		protected int TickLabelSize(Graphics graphics, AxisType axis, int desiredNumberOfTicks, ref double minimum, ref double maximum, out double tickMinimum, out double tickMaximum, out double tickStep)
		{
			string	text;
			int		tickLabelSize;
			double	d;
			double	tickValue;
			SizeF 	textSize;
			
			d = maximum - minimum;
			tickStep = Math.Pow(10.0, Math.Floor(Math.Log10(d / desiredNumberOfTicks)));  // Step size to power of 10, tick labels 0, 1, 2, 3, ...*10^x
			if (d / tickStep > 5 * desiredNumberOfTicks) tickStep *= 5; // Tick labels 0, 5, 10, 15, ...*10^x
			else if (d / tickStep > 2 * desiredNumberOfTicks) tickStep *= 2; // Tick labels 0, 2, 4, 6, ...*10^x
			tickMinimum = tickStep * Math.Floor(2.0 + minimum / tickStep);
			while (Math.Abs(minimum - tickMinimum) > tickStep / 1.9) tickMinimum -= tickStep;
			tickMaximum = tickMinimum;
			while (Math.Abs(maximum - tickMaximum) > tickStep / 1.9) tickMaximum += tickStep;
			if (enforceTickBounds) {	// Range will be readjusted yTickMaximum tick
				if (tickMinimum < minimum) minimum = tickMinimum;
				if (tickMaximum > maximum) maximum = tickMaximum;
			} else	{		// Otherwise
				if (tickMinimum < minimum) tickMinimum += tickStep;
				if (tickMaximum > maximum) tickMaximum -= tickStep;
			}

			tickLabelSize = 0;
			
			for (tickValue = tickMinimum; tickValue < tickMaximum + tickStep / 2; tickValue += tickStep) {
				text = String.Format("{0:g}", (Math.Abs(tickValue) < tickStep / 4.0) ? 0.0 : tickValue);
				textSize = graphics.MeasureString(text, scaleFont);
				tickLabelSize = Math.Max(tickLabelSize, (int)((axis == AxisType.Y) ? textSize.Width : textSize.Height));
			}
			
			return 4 + tickLabelSize;
		}

		protected void DrawFrame(Graphics graphics)
		{
			Rectangle		scaledAxisFrameRectangle;
			Rectangle		xAxisRectangle;
			Rectangle		yAxisRectangle;
			Rectangle		titleRectangle;
			RectangleF		labelRectangle;
			int				xc;
			int 			yc;
			int				temp;
			int				tickLength;
			int				xTickLabelSize;
			int				yTickLabelSize;
			int				bottomCorrection;
			int				leftCorrection;
			double			xTickMinimum;
			double			xTickMaximum;
			double			xTickStep;
			double			yTickMinimum;
			double			yTickMaximum;
			double			yTickStep;
			string			s;
			SizeF			textSize;

			if (xMin == xMax) {
				double x = xMax;
				double p = 0.1;
				if (x > 1E-100) {
					xMax = (1 + p) * x;
					xMin = (1 - p) * x;
				} else if (x < -1E-100) {
					xMax = (1 - p) * x;
					xMin = (1 + p) * x;
				} else {
					xMax = 1;
					xMin = -1;
				}
			}
			if (yMin == yMax) {
				double y = yMax;
				double p = 0.1;
				if (y > 1E-100) {
					yMax = (1 + p) * y;
					yMin = (1 - p) * y;
				} else if (y < -1E-100) {
					yMax = (1 - p) * y;
					yMin = (1 + p) * y;
				} else {
					yMax = 1;
					yMin = -1;
				}
			}
			yTickLabelSize = TickLabelSize(graphics, AxisType.Y, 3, ref yMin, ref yMax, out yTickMinimum, out yTickMaximum, out yTickStep);
			xTickLabelSize = TickLabelSize(graphics, AxisType.X, 3, ref xMin, ref xMax, out xTickMinimum, out xTickMaximum, out xTickStep) + 6;
			textSize = graphics.MeasureString("ABC", scaleFont);
			
			bottomCorrection = !axisX.Title.IsEmpty ? (int)(axisX.Title.Measure(graphics).Height * 1.2) : 0;
			leftCorrection = !axisY.Title.IsEmpty ?  (int)(axisY.Title.Measure(graphics).Height * 1.2) : 0;
			
			scaledAxisFrameRectangle = new Rectangle(
				drawRectangle.Left + leftCorrection,
				drawRectangle.Top,
				drawRectangle.Width - leftCorrection,
				drawRectangle.Height - bottomCorrection
			);
			
			axisFrameRectangle = new Rectangle(
				scaledAxisFrameRectangle.Left + yTickLabelSize,
				scaledAxisFrameRectangle.Top + 1,
				scaledAxisFrameRectangle.Width - yTickLabelSize,
				scaledAxisFrameRectangle.Height - 1 - xTickLabelSize
			);
			
			curveRectangle = new Rectangle(
				axisFrameRectangle.Left + drawRectangle.Width / 40,
				axisFrameRectangle.Top + drawRectangle.Height / 40,
				axisFrameRectangle.Width - drawRectangle.Width / 20,
				axisFrameRectangle.Height - drawRectangle.Height / 20
			);
			
			s = String.Format("{0:g}", xTickMaximum);
			textSize = graphics.MeasureString(s, scaleFont);
			temp = drawRectangle.Right - (int)Math.Floor(0.5 + (curveRectangle.Right - curveRectangle.Left) / (xMax-xMin) * (xTickMaximum - xMin)) - curveRectangle.Left - (int)textSize.Width / 2 - 1;
			if (temp < 0) curveRectangle.Width += temp;
			if ((curveRectangle.Width < 10) || (curveRectangle.Height < 10)) return;
			dx = curveRectangle.Width / (xMax - xMin);
			dy = curveRectangle.Height / (yMax - yMin);
			x0 = curveRectangle.Left;
			y0 = curveRectangle.Bottom;

			graphics.DrawLine(axisPen, axisFrameRectangle.Left, axisFrameRectangle.Top, axisFrameRectangle.Left, axisFrameRectangle.Bottom);
			graphics.DrawLine(axisPen, axisFrameRectangle.Left, axisFrameRectangle.Bottom, axisFrameRectangle.Right, axisFrameRectangle.Bottom);
			graphics.DrawLine(axisPen, axisFrameRectangle.Right, axisFrameRectangle.Bottom, axisFrameRectangle.Right, axisFrameRectangle.Top);
			graphics.DrawLine(axisPen, axisFrameRectangle.Right, axisFrameRectangle.Top, axisFrameRectangle.Left, axisFrameRectangle.Top);

			
			xAxisRectangle = new Rectangle(
				axisFrameRectangle.Left,
				axisFrameRectangle.Bottom,
				drawRectangle.Right - axisFrameRectangle.Left,
				drawRectangle.Bottom - axisFrameRectangle.Bottom
			);
			
			yAxisRectangle = new Rectangle(
				drawRectangle.Left,
				drawRectangle.Top,
				axisFrameRectangle.Left - drawRectangle.Left,
				axisFrameRectangle.Bottom - drawRectangle.Top
			);
			
			tickLength = Math.Max((drawRectangle.Right - drawRectangle.Left) / 100, 3);
			textSize = graphics.MeasureString("8", scaleFont);
			
			double xTickValue = xTickMinimum;
			do {
				xc = (int)Math.Floor(0.5 + x0 + dx * (xTickValue - xMin));
				// Draw grid:
				if (axisX.GridVisible) graphics.DrawLine(gridPen, xc, axisFrameRectangle.Bottom, xc, axisFrameRectangle.Top);
				// Draw ticks:
				graphics.DrawLine(tickPen, xc, axisFrameRectangle.Top, xc, axisFrameRectangle.Top + tickLength);
				graphics.DrawLine(tickPen, xc, axisFrameRectangle.Bottom - 1, xc, axisFrameRectangle.Bottom - 1 - tickLength);
				s = String.Format("{0:g}", (Math.Abs(xTickValue) < xTickStep / 4.0) ? 0.0 : xTickValue);
				textSize = graphics.MeasureString(s, scaleFont);
				labelRectangle = new RectangleF(
					xc - textSize.Width / 2,
					axisFrameRectangle.Bottom + scalePadding,
					textSize.Width,
					textSize.Height
				);
				graphics.DrawString(s, scaleFont, scaleBrush, labelRectangle);
			} while ((xTickValue += xTickStep) < xTickMaximum + xTickStep / 3);
			
			double yTickValue = yTickMinimum;
			do {
				yc = (int)Math.Floor(0.5 + y0 - dy * (yTickValue - yMin));
				if (axisY.GridVisible) graphics.DrawLine(gridPen, axisFrameRectangle.Left, yc, axisFrameRectangle.Right, yc);
				graphics.DrawLine(tickPen, axisFrameRectangle.Left, yc, axisFrameRectangle.Left + tickLength, yc);
				graphics.DrawLine(tickPen, axisFrameRectangle.Right - 1, yc, axisFrameRectangle.Right - 1 - tickLength, yc);
				s = String.Format("{0:g}", (Math.Abs(yTickValue) < yTickStep / 4.0) ? 0.0 : yTickValue);
				textSize = graphics.MeasureString(s, scaleFont);
				labelRectangle = new RectangleF(
					axisFrameRectangle.Left - scalePadding - textSize.Width,
					yc - textSize.Height / 2,
					textSize.Width,
					textSize.Height
				);
				graphics.DrawString(s, scaleFont, scaleBrush, labelRectangle);
			} while ((yTickValue += yTickStep) < yTickMaximum + yTickStep/3);
			
			if (!axisX.Title.IsEmpty) {
				titleRectangle = new Rectangle(
					axisFrameRectangle.Left,
					scaledAxisFrameRectangle.Bottom,
					drawRectangle.Right - axisFrameRectangle.Left,
					drawRectangle.Bottom - scaledAxisFrameRectangle.Bottom
				);
				
				axisX.Title.Draw(graphics, titleRectangle);
			}
			
			if (!axisY.Title.IsEmpty) {
				textSize = axisY.Title.Measure(graphics);
				state = graphics.Save();
				graphics.TranslateTransform(drawRectangle.Left, drawRectangle.Height / 2);
				graphics.RotateTransform(-90);
				axisY.Title.Draw(graphics, 0, textSize.Height / 2.0f);
				graphics.Restore(state);
			}
		}
		
		protected override void OnPaint(PaintEventArgs e)
		{
			e.Graphics.SmoothingMode = lineSmoothingMode;
			e.Graphics.TextRenderingHint = textRenderingHint;
			
			base.OnPaint(e);
			
			UpdateAxisBounds();
			
			if (boundsUndefined) return;
			
			drawRectangle = new Rectangle(
				this.ClientRectangle.Left + this.Padding.Left,
				this.ClientRectangle.Top + this.Padding.Top,
				this.ClientRectangle.Width - this.Padding.Left - this.Padding.Right,
				this.ClientRectangle.Height - this.Padding.Top - this.Padding.Bottom
			);

			DrawFrame(e.Graphics);
			
			Point[] points;
			DataPointList dataPoints;
			
			e.Graphics.IntersectClip(axisFrameRectangle);
			
			foreach (Curve curve in curves) {
				if (curve.Visible) {
					dataPoints = curve.Points;
					
					lock (dataPoints.SyncRoot) {
						points = new Point[dataPoints.Count];
						
						for (int i = 0; i < points.Length; i++) {
							points[i].X = (int)(x0 + dx * (dataPoints[i].X- xMin));
							points[i].Y = (int)(y0 - dy * (dataPoints[i].Y -yMin));
						}
						
						if (dataPoints.Count < 1) continue;
					}
					
					if (curve.HasLine && points.Length > 1) e.Graphics.DrawLines(curve.Line, points);
					
					if (curve.Symbol != null) {
						e.Graphics.SmoothingMode = symbolSmoothingMode;
						foreach (Point point in points) {
							curve.Symbol.Draw(e.Graphics, point);
						}
						e.Graphics.SmoothingMode = lineSmoothingMode;
					}
				}
			}
			
			e.Graphics.ResetClip();
		}
		
		protected override void OnResize(EventArgs e)
		{
			base.OnResize(e);
			this.Refresh();
		}
		
		[System.ComponentModel.EditorBrowsableAttribute()]
		protected override void OnCreateControl()
		{
			base.OnCreateControl();
			if (this.Parent != null) this.BackColor = this.Parent.BackColor;
		}
	}
}