﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: Ouroboros
 * Dátum: 2010.01.09.
 * Idő: 12:39
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace Noise.Controls
{
	/// <summary>
	/// Description of NumericControl.
	/// </summary>
	public partial class NumericControl : UserControl
	{
		protected double minValue = Double.MinValue;
		protected double maxValue = Double.MaxValue;
		protected double value = Double.NaN;
		protected string format = "N";
		
		
		[EditorBrowsable()]
		public event EventHandler ValueChanged;
		
		
		public NumericControl()
		{
			InitializeComponent();
			textBox.Leave += HandleLeaveEvents;
			textBox.PreviewKeyDown += HandlePreviewKeyDownEvents;
		}
		
		
		[EditorBrowsable()]
		public string Format {
			get { return format; }
			set { format = value; }
		}
	
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[EditorBrowsable()]
		public Label Label {
			get { return label; }
		}		
		
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[EditorBrowsable()]
		public TextBox TextBox {
			get { return textBox; }
		}
			
		[EditorBrowsable()]
		public double MinValue {
			get { return minValue; }
			set { minValue = value; }
		}
		
		[EditorBrowsable()]
		public double MaxValue {
			get { return maxValue; }
			set { maxValue = value; }
		}
		
		public double Value {
			get { return value; }
		}
		
		
		public void HandlePreviewKeyDownEvents(object sender, PreviewKeyDownEventArgs e)
		{
			if (sender == textBox) {
				if (e.KeyData == Keys.Enter) {
					ParseValue();
				}
			}
		}
		
		public void HandleLeaveEvents(object sender, EventArgs e)
		{
			if (sender == textBox) {
				ParseValue();
			}
		}
		
		public void SetValue(double value, string naNString)
		{
			this.value = value;
			
			EnsureBounds();
			
			this.TextBox.Text = GetText(naNString);
		}
		
		public void SetValue(double value)
		{
			SetValue(value, String.Empty);
		}
		
		protected virtual void OnValueChanged(EventArgs e)
		{
			if (ValueChanged != null) {
				ValueChanged(this, e);
			}
		}
		
		protected string GetText(string naNString)
		{
			if (Double.IsNaN(value)) {
				return naNString;
			} else {
				string text;

				try {
					text = value.ToString(format);
				} catch {
					text = value.ToString();
				}
				
				return text;
			}
		}
		
		protected void EnsureBounds()
		{
			if (this.value < minValue) this.value = minValue;
			if (this.value > maxValue) this.value = maxValue;
		}
		
		protected void ParseValue()
		{
			double oldValue = value;
			double parsed = Double.NaN;
			
			if (Double.TryParse(textBox.Text, out parsed)) {
				value = parsed;
				
				EnsureBounds();
				
				textBox.Text = GetText(String.Empty);
				if (value != oldValue) OnValueChanged(new EventArgs());
			} else {
				SetValue(Double.NaN, String.Empty);
			}
		}
	}
}
