﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2010.01.19.
 * Idő: 11:00
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
//
using Noise.DataAcquisition;

namespace Noise.Controls
{
	public enum SensorEditorMode {
		Calibrate,
		Edit
	}
	
	/// <summary>
	/// Description of SensorEditor.
	/// </summary>
	public partial class SensorEditor : Form
	{
		protected Sensor sensor;
		protected Edaq530 device;
		protected int channelIndex = 0;
		//
		protected SensorEditorMode mode;
		protected int lastGraphWidth;
		protected Timer timer;
		List<double> voltageValues;
		List<double> quantityValues;
		
		
		
		public SensorEditor(SensorEditorMode mode, Sensor sensor, Edaq530 device, int channelIndex)
		{
			InitializeComponent();
			//
			modeComboBox.SelectedIndex = (mode == SensorEditorMode.Calibrate) ? 0 : 1;
			modeComboBox.SelectionChangeCommitted += HandleSelectionChangeCommittedEvents;
			channelLabel.Text = Edaq530.ChannelNames[channelIndex];
			sensorControl1.Sensor = sensor;
			sensorControl1.SensorRecreated += HandleSensorRecreatedEvents;
			dataGridView1.CellValueChanged += HandleCellValueChangedEvents;
			//
			this.sensor = sensor;
			this.device = device;
			this.channelIndex = channelIndex;
			ChangeMode(mode);
		}
		
		public SensorEditor() : this(SensorEditorMode.Edit, null, null, 0) { }
		
		
		public Edaq530 Device {
			get { return device; }
			set { device = value; }
		}
		
		public Sensor Sensor {
			get { return sensor; }
		}
		
		
		public void ChangeMode(SensorEditorMode mode)
		{
			TableLayoutColumnStyleCollection columns = tableLayoutPanel1.ColumnStyles;
			
			this.mode = mode;
			SetDataGridView();
			UpdateGraph();
			
			if (mode == SensorEditorMode.Calibrate) {
				SetClientSizeCore((int)columns[1].Width + lastGraphWidth, ClientSize.Height);
				columns[0].SizeType = SizeType.Percent;
				columns[0].Width = 100F;
				this.FormBorderStyle = FormBorderStyle.Sizable;
				this.Location = new Point(0, 0);
				// Enable controls:
				label1.Enabled = true;
				addPointButton.Enabled = true;
				removePointButton.Enabled = true;
			} else if (mode == SensorEditorMode.Edit) {
				lastGraphWidth = graph.Width;
				if (this.WindowState == FormWindowState.Maximized) {
					this.WindowState = FormWindowState.Normal;
				}
				SetClientSizeCore((int)columns[1].Width, ClientSize.Height);
				columns[0].SizeType = SizeType.Absolute;
				columns[0].Width = 0;
				this.FormBorderStyle = FormBorderStyle.FixedToolWindow;
				// Disable controls:
				label1.Enabled = false;
				addPointButton.Enabled = false;
				removePointButton.Enabled = false;
			}
		}
		
		
		protected void HandleCellValueChangedEvents(object sender, DataGridViewCellEventArgs e)
		{
			if (sender == dataGridView1) {
				if (e.ColumnIndex == 0) return;
				if (sensor == null) return;
				if (mode == SensorEditorMode.Calibrate) {
					return;	// This will be handled by button clicks.
				} else if (mode == SensorEditorMode.Edit) {
					double oldValue = Double.NaN;
					double newValue = Double.NaN;
					bool isSuccessful = false;
					
					// Obtain the new value:
					try {
						newValue =
							Convert.ToDouble(dataGridView1.Rows[e.RowIndex].Cells[e.ColumnIndex].Value);
						isSuccessful = (dataGridView1.Rows[e.RowIndex].Cells[e.ColumnIndex].Value != null);
					} catch {
						isSuccessful = false;
					}
					
					if (sensor is LinearSensor) {
						LinearSensor linear = sensor as LinearSensor;
						
						if (e.RowIndex == 0) {
							oldValue = linear.Intercept;
							if (isSuccessful) {
								linear.Intercept = newValue;
							} 
						} else if (e.RowIndex == 1) {
							oldValue = linear.Slope;
							if (isSuccessful) {
								linear.Slope = newValue;
							}
						}
					} else if (sensor is Thermistor) {
						Thermistor thermistor = sensor as Thermistor;
						
						if (e.RowIndex == 0) {
							oldValue = thermistor.ReferenceResistance;
							if (isSuccessful) {
								thermistor.ReferenceResistance = newValue;
							}
						} else if (e.RowIndex == 1) {
							oldValue = thermistor.Coefficient;
							if (isSuccessful) {
								thermistor.Coefficient = newValue;
							}
						} else if (e.RowIndex == 2) {
							oldValue = thermistor.ProbeResistance;
							if (isSuccessful) {
								thermistor.ProbeResistance = newValue;
							}
						}
					}
					// TODO: Implement for other sensor types.
					if (isSuccessful) {
						sensorControl1.Refresh();
					} else {
						dataGridView1.Rows[e.RowIndex].Cells[e.ColumnIndex].Value = oldValue;
					}
				}
			}
		}
		
		protected void HandleClickEvents(object sender, EventArgs e)
		{
			if (sender == okButton) {
				this.DialogResult = DialogResult.OK;
			} else if (sender == cancelButton) {
				this.DialogResult = DialogResult.Cancel;
			} else if (sender == addPointButton) {
				bool isSuccessful = false;
				double quantityValue = Double.NaN;
				double voltageValue = Double.NaN;
				
				try {
					voltageValue = Convert.ToDouble(
						dataGridView1.Rows[dataGridView1.Rows.Count - 1].Cells[0].Value);
					quantityValue = Convert.ToDouble(
						dataGridView1.Rows[dataGridView1.Rows.Count - 1].Cells[1].Value);
					isSuccessful = (dataGridView1.Rows[dataGridView1.Rows.Count - 1].Cells[1].Value != null);
				} catch {
					isSuccessful = false;
				}
				
				if (isSuccessful) {
					voltageValues.Add(voltageValue);
					quantityValues.Add(quantityValue);
					dataGridView1.Rows[dataGridView1.Rows.Count - 1].ReadOnly = true;
					dataGridView1.Rows.Add();
					if (voltageValues.Count > 1) {
						sensor.Calibrate(voltageValues, quantityValues);
						sensorControl1.Refresh();
					}
					UpdateGraph();
				} else {
					dataGridView1.Rows[dataGridView1.Rows.Count - 1].Cells[1].Value = null;
				}
			} else if (sender == removePointButton) {
				DataGridViewSelectedCellCollection cells = dataGridView1.SelectedCells;
				if (cells == null) return;
				
				foreach (DataGridViewCell cell in cells) {
					if (cell.RowIndex >= voltageValues.Count) continue;
					voltageValues.RemoveAt(cell.RowIndex);
					quantityValues.RemoveAt(cell.RowIndex);
					dataGridView1.Rows.RemoveAt(cell.RowIndex);
					// TODO: This may be dangerous, check if it works.
				}
				
				if (voltageValues.Count > 1) {
					sensor.Calibrate(voltageValues, quantityValues);
					sensorControl1.Refresh();
				}
				UpdateGraph();
			}
		}
		
		protected void HandleSelectionChangeCommittedEvents(object sender, EventArgs e)
		{
			if (sender == modeComboBox) {
				switch (modeComboBox.SelectedIndex) {
					case 0:
						ChangeMode(SensorEditorMode.Calibrate);
						break;
					case 1:
					default:
						ChangeMode(SensorEditorMode.Edit);
						break;
				}
			}
		}
		
		protected void HandleSensorRecreatedEvents(object sender, EventArgs e)
		{
			if (sender == sensorControl1) {
				sensor = sensorControl1.Sensor;
				SetDataGridView();
				UpdateGraph();
			}
		}
		
		protected void HandleVoltageUpdatedEvents(object sender, EventArgs e)
		{
			if (device == null) return;
			double[] measuredValues = device.Measure(1);
			dataGridView1.Rows[dataGridView1.Rows.Count - 1].Cells[0].Value = measuredValues[channelIndex];
		}
		
		[System.ComponentModel.EditorBrowsableAttribute()]
		protected override void OnFormClosing(FormClosingEventArgs e)
		{
			if (timer != null) timer.Stop();
			base.OnFormClosing(e);
		}
		
		protected void SetDataGridView()
		{
			int columnWidth = (dataGridView1.ClientSize.Width - 3) / 2;
			int widthDifference = 25;
			
			if (timer != null) timer.Stop();
			dataGridView1.Rows.Clear();
			dataGridView1.Columns.Clear();
			dataGridView1.Enabled = true;
			if (voltageValues == null) {
				voltageValues = new List<double>();
			} else {
				voltageValues.Clear();
			}
			if (quantityValues == null) {
				quantityValues = new List<double>();
			} else {
				quantityValues.Clear();
			}
			
			if (mode == SensorEditorMode.Calibrate) {
				dataGridView1.SelectionMode = DataGridViewSelectionMode.CellSelect;
				dataGridView1.Columns.Add("VoltageColumn", "Voltage [V]");
				dataGridView1.Columns[0].ReadOnly = true;
				dataGridView1.Columns[0].Width = columnWidth;
				dataGridView1.Columns[0].DefaultCellStyle.Format = "g3";
				//
				dataGridView1.Columns.Add("QuantityColumn", String.Empty);
				dataGridView1.Columns[1].ReadOnly = false;
				dataGridView1.Columns[1].Width = columnWidth;
				//
				if (sensor == null) {
					dataGridView1.Rows.Add();
					dataGridView1.Rows[0].Cells[0].Value = "No sensor defined.";
					dataGridView1.Enabled = false;
					return;
				}
				//
				if (String.IsNullOrEmpty(sensor.Quantity)) {
					dataGridView1.Columns[1].HeaderText = "Sensor voltage [V]";
				} else {
					dataGridView1.Columns[1].HeaderText = sensor.Quantity;
					if (!String.IsNullOrEmpty(sensor.Unit)) {
						dataGridView1.Columns[1].HeaderText += String.Format(" [{0}]", sensor.Unit);
					}
				}
				dataGridView1.Rows.Add();
				if (device == null) {
					dataGridView1.Rows[0].Cells[0].Value = "No device defined.";
					dataGridView1.Enabled = false;
				} else {
					timer = new Timer();
					timer.Interval = 300;
					timer.Tick += HandleVoltageUpdatedEvents;
					timer.Start();
				}
			} else if (mode == SensorEditorMode.Edit) {
				dataGridView1.AllowUserToAddRows = false;
				dataGridView1.Columns.Add("ParameterColumn", "Parameter");
				dataGridView1.Columns[0].ReadOnly = true;
				dataGridView1.Columns[0].Width = columnWidth + widthDifference;
				//
				dataGridView1.Columns.Add("ValueColumn", "Value");
				dataGridView1.Columns[1].ReadOnly = false;
				dataGridView1.Columns[1].Width = columnWidth - widthDifference;
				//
				if (sensor == null) {
					dataGridView1.Rows.Add();
					dataGridView1.Rows[0].Cells[0].Value = "No sensor defined.";
					dataGridView1.Enabled = false;
					return;
				}
				//
				if (sensor is LinearSensor) {
					LinearSensor linear = sensor as LinearSensor;
					dataGridView1.Rows.Add("Intercept" + (String.IsNullOrEmpty(sensor.Unit) ? String.Empty :
					                                      String.Format(" [{0}]", sensor.Unit)), linear.Intercept);
					dataGridView1.Rows.Add("Slope" + (String.IsNullOrEmpty(sensor.Unit) ? " [1/V]" :
					                                      String.Format(" [({0})/V]", sensor.Unit)), linear.Slope);
				} else if (sensor is Thermistor) {
					Thermistor thermistor = sensor as Thermistor;
					dataGridView1.Rows.Add("Reference resistance [Ω]", thermistor.ReferenceResistance);
					dataGridView1.Rows.Add("B coefficient [K]", thermistor.Coefficient);
					dataGridView1.Rows.Add("Probe resistance [Ω]", thermistor.ProbeResistance);
				}
				// TODO: Add other sensor types.
			}
		}
	
		protected void UpdateGraph()
		{
			int length = Math.Min(voltageValues.Count, quantityValues.Count);
			int calculationPoints = 1024;
			double minX = Double.MaxValue;
			double maxX = Double.MinValue;
			double dX;
			Curve measured = new Curve(length);
			Curve calculated = new Curve(calculationPoints);
			
			graph.Curves.Clear();
			graph.AxisX.Title.Text = "Voltage [V]";
			graph.AxisY.Title.Text	= (sensor != null) ? sensor.GetLabel() : String.Empty;
			
			for (int i = 0; i < length; i++) {
				measured.Points.Add(voltageValues[i], quantityValues[i]);
				if (voltageValues[i] < minX) minX = voltageValues[i];
				if (voltageValues[i] > maxX) maxX = voltageValues[i];
			}
			
			measured.Line.Width = 0.0f;
			measured.Symbol = Symbol.Circle;
			measured.Symbol.Size = 10.0f;
			measured.Symbol.Border.Color = Color.Gray;
			
			graph.Curves.Add(measured);
			
			if (length > 1) {
				dX = (maxX - minX) / (double)(calculationPoints - 1);
				for (int i = 0; i < calculationPoints; i++) {
					calculated.Points.Add(minX + i * dX, sensor.GetValue(minX + i * dX));
				}
				
				calculated.Symbol = null;
				calculated.Line.Width = 3.0f;
				calculated.Line.Color = Color.DarkOrange;
				graph.Curves.Add(calculated);
			}
			
			graph.Refresh();
		}
	}
}
