﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.06.22.
 * Idő: 15:40
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.IO.Ports;
using System.Text;
using System.Threading;
//
using TetheredSun.Core;

namespace Noise.DataAcquisition
{
	public class Edaq530 : ISerialDataCollector
	{
		#region CONSTANTS
		public const string CommandToken = "@";
		public const string DeviceInfoCommand = "@I";
		public const string MeasureCommand = "@M";
		public const string MeasureSeriesCommand = "@s";
		public const string SamplingFrequencyCommand = "@f";
		public const string StartSamplingCommand = "@S";
		public const string QueryTimingCommand = "@Q";
		public const double MinSamplingFrequency = 11.0;
		public const double MinVoltage = 0.0;
		public const double MaxVoltage = 5.0;
		public const int BaudRate = 230400;
		public const int AdcBits = 12;
		public const Handshake Handshake = System.IO.Ports.Handshake.None;
		#endregion
		#region FIELDS
		protected int counter = 0;
		protected int decimation = 1;
		protected double resolution;	// 2^adcBits;
		protected double quantum;
		protected double? samplingFrequency = null;		// A per-channel sampling frequency.
		protected Averages averages;
		protected EventWaitHandle ready = new ManualResetEvent(true);
		protected EventWaitHandle samplingStarted = new ManualResetEvent(false);
		protected EventWaitHandle samplingStopped = new ManualResetEvent(true);
		protected ChannelList channels = null;
		protected ISerialPort port;
		protected BackgroundWorker worker;
		protected static List<string> channelNames = new List<string>(new string[] { "A", "B", "C" });
		#endregion
		#region EVENTS
		public event AsyncCompletedEventHandler StoppedOnError;
		#endregion
		
		#region CONSTRUCTORS
		public Edaq530(ISerialPort port)
		{
			this.port = port;
			resolution = Math.Pow(2.0, AdcBits);
			quantum = (MaxVoltage - MinVoltage) / resolution;
			if ((port != null) && port.IsOpen) {
				GetSamplingParameters();
			}
		}	
		#endregion
		
		#region PROPERTIES
		public static List<string> ChannelNames {
			get { return channelNames; }
		}
		
		public Averages Averages {
			get {
				CheckContinuousMode();
				CheckIfReady();
				ready.Reset();
				//
				try {
					GetSamplingParameters();
				} catch {
					averages = null;
				}
				//
				ready.Set();
				
				return averages;
			}
		}
		
		public ChannelList Channels {
			get { return channels; }
			set { channels = value; }
		}
		
		public string Name {
			get { return "Edaq530"; }
		}
		
		public ISerialPort Port {
			get { return port; }		
		}
		
		/// <summary>
		/// Gets the per-channel sampling frequency. The sampling frequency of the A/D converter
		/// is thrice this value, but it takes three cycles of conversion to sample the three channels.
		/// </summary>
		public Nullable<double> SamplingFrequency {
			get { 
				CheckContinuousMode();
				CheckIfReady();
				ready.Reset();
				//
				try {
					GetSamplingParameters();
				} catch {
					samplingFrequency = null;
				}
				//
				ready.Set();
				
				return samplingFrequency / decimation;
			}
		}
		
		public EventWaitHandle SamplingStarted {
			get { return samplingStarted; }
		}
		
		public EventWaitHandle SamplingStopped {
			get { return samplingStopped; }
		}
		#endregion
		
		#region PUBLIC METHODS
		public void Escape()
		{
			Port.Write(new byte[] { 27 });
			Thread.Sleep(600);
		}
		
		public string GetDeviceInfo()
		{
			CheckContinuousMode();
			CheckIfReady();
			ready.Reset();
			//
			string deviceInfo = String.Empty;
			byte[] commandBytes = ConvertToBytes(DeviceInfoCommand, null);
			Encoding ascii = Encoding.ASCII;
			bool continueReading = true;
			string command = "§";
			byte[] cb = ConvertToBytes(command, null);
			byte[] read;
			//
			Send(commandBytes);
			while (continueReading) {
				Port.Write(cb);
				read = Port.Read(1);
				continueReading = (read[0] != cb[0]);
				if (continueReading) {
					deviceInfo += ascii.GetString(read);
				}
			}
			//
			ready.Set();
			return deviceInfo;
		}
		
		public double[] Measure(byte averages)
		{
			CheckContinuousMode();
			CheckIfReady();
			ready.Reset();
			//
			byte[] commandBytes = ConvertToBytes(MeasureCommand, new byte[] { averages > 0 ? averages : (byte)1 });
			byte[] dataBytes;
			double[] measuredVoltages;
			int blockByteCount = 6;	// 3 channels, 2 bytes per channel.
			//
			Send(commandBytes);
			dataBytes = Port.Read(blockByteCount);
			measuredVoltages = ConvertToVoltages(dataBytes);
			//
			ready.Set();
			return measuredVoltages;
		}
		
		public void Reset()
		{
			if ((Port != null) && Port.IsOpen) {
				ready.Reset();
				//
				Escape();
				Port.DiscardOutBuffer();
				Port.DiscardInBuffer();
				//
				ready.Set();
			} else {
				ready.Reset();
			}
		}
		
		public void Send(string command, byte[] parameters)
		{
			CheckContinuousMode();
			CheckIfReady();
			ready.Reset();
			//
			Send(ConvertToBytes(command, parameters));
			//
			ready.Set();
		}
		
		public void Send(string command)
		{
			CheckContinuousMode();
			CheckIfReady();
			ready.Reset();
			//
			Send(ConvertToBytes(command, null));
			//
			ready.Set();
		}
		
		/// Sets the per-channel sampling frequency. The sampling frequency of the A/D converter
		/// is thrice this value, but it takes three cycles of conversion to sample the three channels.
		/// <param name="samplingFrequency">The sampling frequency per channel.</param>
		/// <param name="averages">The number of averages the hardware takes.</param>
		public void SetSamplingFrequency(double samplingFrequency, Averages averages)
		{
			CheckContinuousMode();
			CheckIfReady();
			ready.Reset();
			//
			ushort f;
			byte[] paramBytes = new byte[3];
			byte[] commandBytes;
			int remainder;
			//
			if (samplingFrequency < MinSamplingFrequency) {
				decimation = (int)Math.Ceiling(MinSamplingFrequency / samplingFrequency);
				if (decimation < 1) decimation = 1;
			} else {
				decimation = 1;
			}
			f = (ushort)(3.0 * samplingFrequency * decimation);
			//
			paramBytes[0] = averages.Code;
			paramBytes[1] = (byte)Math.DivRem(f, 256, out remainder);
			paramBytes[2] = (byte)remainder;
			commandBytes = ConvertToBytes(SamplingFrequencyCommand, paramBytes);
			//
			
			//
			try {
				Send(commandBytes);
				GetSamplingParameters();
			} catch (Exception exception) {
				this.samplingFrequency = null;
				throw exception;
			}
			//
			ready.Set();
		}
		
		public void StartSampling(double samplingFrequency, Averages averages)
		{
			SetSamplingFrequency(samplingFrequency, averages);
			StartSampling();
		}
		
		public void StartSampling()
		{
			if (!samplingFrequency.HasValue) {
				Exception exception = new InvalidOperationException("Sampling frequency is not set.");
				throw exception;
			}
			if (channels == null) {
				Exception exception = new InvalidOperationException("No data socket defined.");
				throw exception;
			}
			CheckContinuousMode();
			CheckIfReady();
			ready.Reset();	// Will be set after the BackgroundWorker completes work.
			//
			byte[] commandBytes = ConvertToBytes(StartSamplingCommand, null);
			//
			Send(commandBytes);
			worker = new BackgroundWorker();
			worker.WorkerSupportsCancellation = true;
			worker.DoWork += Sample;
			worker.RunWorkerCompleted += HandleRunWorkerCompletedEvents;
			worker.RunWorkerAsync();
			samplingStarted.Set();
			samplingStopped.Reset();
			counter = 0;
		}
		
		public void StopSampling()
		{
			if ((worker != null) && worker.IsBusy) {
				worker.CancelAsync();
			}
		}
		#endregion
		
		#region PROTECTED METHODS
		protected void CheckContinuousMode()
		{
			if (samplingStarted.WaitOne(0)) {
				Exception exception = new InvalidOperationException("Operation cannot be executed whilst the device is in continuous mode.");
				throw exception;
			}
		}
		
		protected void CheckIfReady()
		{
			if (!ready.WaitOne(1000)) {
				Exception exception = new InvalidOperationException("Timeout in waiting for the device to be ready.");
				throw exception;
			}
		}
			
		protected byte[] ConvertToBytes(string command, byte[] parameters)
		{
			byte[] commandBytes = Encoding.ASCII.GetBytes(command);
			byte[] bytesToSend = (parameters != null) ? new byte[commandBytes.Length + parameters.Length] : new byte[commandBytes.Length];
			Array.Copy(commandBytes, bytesToSend, commandBytes.Length);
			if (parameters != null) {
				Array.Copy(parameters, 0,  bytesToSend, commandBytes.Length, parameters.Length);
			}
			
			return bytesToSend;
		}
		
		protected double[] ConvertToVoltages(byte[] bytes)
		{
			double[] voltages = new double[bytes.Length / 2];
			int v;	// Voltage as 16-bit code.
			
			for (int i = 0; i < voltages.Length; i++) {
				v = bytes[2 * i] << 8;	// TODO: This code assumes 2-byte words, will not work for other word lengths.
				v += bytes[2 * i + 1];
				voltages[i] = (MinVoltage + quantum * v) / averages.Number;
			}
			
			return voltages;
		}
		
		protected void EnqueueReadBlocks(byte[] readBlocks)
		{
			if (channels == null) return;
			if (channels.Count < 1) return;
			//
			byte[] block;
			double[] voltages;
			int blockByteCount = 6;	// 3 channels, 2 bytes per channel.
			int channelCount;
			
			for (int i = 0; i <= readBlocks.Length - blockByteCount; i += blockByteCount) {
				counter++;
				if (counter < decimation) {	// We do not have to enqueue the data.
					continue;
				} 
				counter = 0;
				block = new byte[blockByteCount];
				for (int j = 0; j < block.Length; j++) {
					block[j] = readBlocks[i + j];
				} 
				voltages = ConvertToVoltages(block);
			
				channelCount = Math.Min(voltages.Length, channels.Count);
				for (int k = 0; k < channelCount; k++) {
					lock (channels[k].SyncRoot) {
						channels[k].Add(voltages[k]);
					}
				}
			}
		}
		
		protected void GetSamplingParameters()
		{
			byte[] timing = new byte[3];
			
			Send(ConvertToBytes(QueryTimingCommand, null));
			
			timing = Port.Read(3);
			
			this.averages = Averages.FromCode((byte)(timing[2] & (byte)3));
			samplingFrequency = (24500000.0 / 12) / (65536.0 - (timing[0] * 256 + timing[1])) / averages.Number / 3.0;
		}
		
		protected void HandleRunWorkerCompletedEvents(object sender, RunWorkerCompletedEventArgs e)
		{
			Reset();	// Also sets 'ready'.
			AsyncCompletedEventArgs completedArgs = new AsyncCompletedEventArgs(e.Error, true, null);
			if (e.Error != null) {
				OnStoppedOnError(completedArgs);
				ExceptionReporter.Show(e.Error, true);
			}
		}
		
		protected virtual void OnStoppedOnError(AsyncCompletedEventArgs e)
		{
			if (StoppedOnError != null) {
				StoppedOnError(this, e);
			}
		}
		
		protected void Sample(object sender, DoWorkEventArgs e)
		{
			BackgroundWorker theWorker = sender as BackgroundWorker;
			byte[] readBytes;
			int blockByteCount = 6;	// 3 channels, 2 bytes per channel = 1 block.
			int blockCount = 1;
	
			try {				
				while (!theWorker.CancellationPending) {
					lock (port.SyncRoot) {
						readBytes = port.Read(blockCount * blockByteCount);	// We depend on the ReadTimeout of the port.
					}
					EnqueueReadBlocks(readBytes);
				}
				// Notify the device to exit continuous sampling mode.
				Escape();
				// Read the remaining data:
				lock (port.SyncRoot) {
					readBytes = port.Read();
				}
				if (readBytes == null) return;
				if (readBytes.Length >= blockByteCount) EnqueueReadBlocks(readBytes);
			} catch (Exception exception) {
				Escape();
				exception.Data.Add("Bytes at port", port.BytesToRead);
				throw exception;
			} finally {
				SignalStop();
			}
		}
	
		protected void Send(byte[] bytes)
		{
			byte[] read;
			foreach (byte b in bytes) {
				Port.Write(new byte[] { b });
				read = Port.Read(1);
				if (read[0] != b) {
					Exception exception = new IOException("Sent and received bytes do not match.");
					exception.Data.Add("Sent byte", b);
					exception.Data.Add("Received byte", read[0]);
					throw exception;
				}
			}
		}
		
		protected void SignalStop()
		{
			samplingStarted.Reset();
			samplingStopped.Set();
		}
		#endregion
	}
}
