﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2008.08.11.
 * Idő: 15:03
 * Verzió: 0.0.0
 * Legutóbbi változtatás időpontja:
 * Változtatások:
 * Teendők:
 */

using System;
using System.IO;
using System.IO.Ports;
using System.Windows.Forms;
using FTD2XX_NET;
// Aliases for the disgusting FTDI names:
using FtdiDataBits = FTD2XX_NET.FTDI.FT_DATA_BITS;
using FtdiFlowControl = FTD2XX_NET.FTDI.FT_FLOW_CONTROL;
using FtdiInfoNode = FTD2XX_NET.FTDI.FT_DEVICE_INFO_NODE;
using FtdiParity = FTD2XX_NET.FTDI.FT_PARITY;
using FtdiPurgeMask = FTD2XX_NET.FTDI.FT_PURGE;
using FtdiStatus = FTD2XX_NET.FTDI.FT_STATUS;
using FtdiStopBits = FTD2XX_NET.FTDI.FT_STOP_BITS;

namespace Noise.DataAcquisition
{
	/// <summary>
	/// Description of Ftdi.
	/// </summary>
	public class Ftdi : FTDI, ISerialPort
	{
		protected int baudRate = 57600;
		protected int dataBits = 8;
		protected int readTimeout = 1000;
		protected int writeTimeout = 1000;
		protected string description = null;
		protected string serialNumber = null;
		protected Handshake handshake = Handshake.None;
		protected Parity parity = Parity.None;
		protected StopBits stopBits = StopBits.Two;
		protected FtdiStatus status = FtdiStatus.FT_OK;
		protected object syncRoot = new Object();
		
		public Ftdi() : base() { }
		
		public Ftdi(string serialNumber) : base()
		{
			this.serialNumber = serialNumber;
		}

		public int BaudRate {
			get {
				return baudRate;
			}
			set {
				baudRate = value;
				status = SetBaudRate((uint)baudRate);
				CheckStatus(true);
			}
		}
		
		public int BytesToRead {
			get {
				uint bytesAvailable = 0;
				status = GetRxBytesAvailable(ref bytesAvailable);
				CheckStatus(true);
				return (int)bytesAvailable;
			}
		}
		
		public int BytesToWrite {
			get {
				uint bytesWaiting = 0;
				status = GetTxBytesWaiting(ref bytesWaiting);
				CheckStatus(true);
				return (int)bytesWaiting;
			}
		}
		
		public int DataBits {
			get {
				return dataBits;
			}
			set {
				dataBits = value;
				SetTransmission();
			}
		}
		
		public string Description {
			get {
				if (IsOpen) {
					status = GetDescription(out description);
					CheckStatus(true);
				} else {
					description = null;
				}
				return description;
			}
		}
		
		public Handshake Handshake {
			get {
				return handshake;
			}
			set {
				ushort flowControl;
				handshake = value;
				
				switch (handshake) {
					case Handshake.None:
						flowControl = FtdiFlowControl.FT_FLOW_NONE;
						break;
					case Handshake.RequestToSend:
						flowControl = FtdiFlowControl.FT_FLOW_RTS_CTS;
						break;
					case Handshake.RequestToSendXOnXOff:
						throw new InvalidOperationException("RequestToSendXOnXOff mode is not supported in this class.");
					case Handshake.XOnXOff:
						flowControl = FtdiFlowControl.FT_FLOW_XON_XOFF;
						break;
					default:
						flowControl = FtdiFlowControl.FT_FLOW_NONE;
						break;
				}
				
				status = SetFlowControl(flowControl, 0, 0);	// TODO: Check if it is OK to select 0 as XOn / XOff signaller byte.
				CheckStatus(true);
			}
		}
		
		public int ID {
			get {
				if (IsOpen) {
					uint id = 0;
					status = GetDeviceID(ref id);
					CheckStatus(true);
					return (int)id;
				} else {
					return -1;
				}
			}
		}
		
		public Parity Parity {
			get {
				return parity;
			}
			set {
				parity = value;
				SetTransmission();
			}
		}
		
		public int ReadTimeout {
			get {
				return readTimeout;
			}
			set {
				readTimeout = value;
				status = SetTimeouts((uint)readTimeout, (uint)writeTimeout);
				CheckStatus(true);
			}
		}
		
		public string SerialNumber {
			get {
				if (IsOpen) {
					status = GetSerialNumber(out serialNumber);
					CheckStatus(true);
				} else {
					serialNumber = null;
				}
				return serialNumber;
			} set { 
				serialNumber = value;
			}
		}
		
		public FtdiStatus Status {
			get { return status; }
		}
		
		public StopBits StopBits {
			get {
				return stopBits;
			}
			set {
				stopBits = value;
				SetTransmission();
			}
		}
		
		public object SyncRoot {
			get { return syncRoot; }
		}
		
		public int WriteTimeout {
			get {
				return writeTimeout;
			}
			set {
				writeTimeout = value;
				status = SetTimeouts((uint)readTimeout, (uint)writeTimeout);
				CheckStatus(true);
			}
		}
		
		public bool CheckStatus(bool throwException)
		{
			bool isOK = (status == FtdiStatus.FT_OK);
			string statusString = status.ToString().Replace('_', '-');
			
			if (!isOK && throwException) {
				Exception exception = new IOException("Ftdi error: " + statusString);
				throw exception;
			}
			
			return isOK;
		}
		
		void IPort.Close()
		{
			status = base.Close();
			CheckStatus(true);
		}
				
		public void DiscardInBuffer()
		{
			status = Purge(FtdiPurgeMask.FT_PURGE_RX);
			CheckStatus(true);
		}
		
		public void DiscardOutBuffer()
		{
			status = Purge(FtdiPurgeMask.FT_PURGE_TX);
			CheckStatus(true);
		}
		
		public static string[] GetDescriptions()
		{
			uint deviceCount = 0;
			string[] descriptions = null;
			FtdiInfoNode[] deviceInfo;
			FTDI lister = new FTDI();
			FtdiStatus listerStatus = lister.GetNumberOfDevices(ref deviceCount);
			
			if (deviceCount > 0) {
				descriptions = new string[deviceCount];
				deviceInfo = new FtdiInfoNode[deviceCount];
				listerStatus = lister.GetDeviceList(deviceInfo);
				
				for (int i = 0; i < deviceCount; i++) {
					descriptions[i] = deviceInfo[i].Description;
				}
			}
			
			return descriptions;
		}
		
		public static string[] GetSerialNumbers()
		{
			uint deviceCount = 0;
			string[] serialNumbers = null;
			FtdiInfoNode[] deviceInfo;
			FTDI lister = new FTDI();
			FtdiStatus listerStatus = lister.GetNumberOfDevices(ref deviceCount);
			
			if (deviceCount > 0) {
				serialNumbers = new string[deviceCount];
				deviceInfo = new FtdiInfoNode[deviceCount];
				listerStatus = lister.GetDeviceList(deviceInfo);
				
				for (int i = 0; i < deviceCount; i++) {
					serialNumbers[i] = deviceInfo[i].SerialNumber;
				}
			}
			
			return serialNumbers;
		}
		
		public void Initialise()
		{
			BaudRate = baudRate;
			Handshake = handshake;
			ReadTimeout = readTimeout;
			WriteTimeout = writeTimeout;
			SetTransmission();
		}
		
		public void Open()
		{
			if (!String.IsNullOrEmpty(serialNumber)) {
				status = OpenBySerialNumber(serialNumber);
			} else {
				status = OpenByIndex(0);
			}
			CheckStatus(true);
		}
		
		public byte[] Read(int count)
		{
			byte[] bytesRead = new byte[count];
			uint numberOfBytesRead = 0;
			status = Read(bytesRead, (uint)count, ref numberOfBytesRead);
			CheckStatus(true);
			if (numberOfBytesRead < count) {
				throw new IOException("Could not read the specified number of bytes.");
			}
			return bytesRead;
		}
		
		public byte[] Read()
		{
			byte[] bytesRead;
			uint numberOfBytesAvailable = 0;
			uint numberOfBytesRead = 0;
			status = GetRxBytesAvailable(ref numberOfBytesAvailable);
			CheckStatus(true);
			if (numberOfBytesAvailable > 0) {
				bytesRead = new byte[numberOfBytesAvailable];
				status = Read(bytesRead, numberOfBytesAvailable, ref numberOfBytesRead);
				return bytesRead;
			} else {
				return null;
			}
		}
		
		public void SetTransmission(int dataBits, StopBits stopBits, Parity parity)
		{
			byte ftdiDB = (dataBits == 7) ? FtdiDataBits.FT_BITS_7 : FtdiDataBits.FT_BITS_8;
			byte ftdiSB = (stopBits == StopBits.One) ? FtdiStopBits.FT_STOP_BITS_1 : FtdiStopBits.FT_STOP_BITS_2;
			byte ftdiP;
			
			this.dataBits = dataBits;
			this.stopBits = stopBits;
			this.parity = parity;
			
			switch (parity) {
				case Parity.Even:
					ftdiP = FtdiParity.FT_PARITY_EVEN;
					break;
				case Parity.Mark:
					ftdiP = FtdiParity.FT_PARITY_MARK;
					break;
				case Parity.None:
					ftdiP = FtdiParity.FT_PARITY_NONE;
					break;
				case Parity.Odd:
					ftdiP = FtdiParity.FT_PARITY_ODD;
					break;
				case Parity.Space:
					ftdiP = FtdiParity.FT_PARITY_SPACE;
					break;
				default:
					ftdiP = FtdiParity.FT_PARITY_NONE;
					break;
			}
			
			status = SetDataCharacteristics(ftdiDB, ftdiSB, ftdiP);
			CheckStatus(true);
		}
		
		public void SetTransmission()
		{
			SetTransmission(dataBits, stopBits, parity);
		}
		
		public void Write(byte[] buffer)
		{
			uint numberOfBytesWritten = 0;
			status = Write(buffer, buffer.Length, ref numberOfBytesWritten);
			CheckStatus(true);
		}
		
	}
}
