﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2010.01.08.
 * Idő: 12:59
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections.Generic;
using System.Xml;
//
using TetheredSun.Core.Mathematics;

namespace Noise.DataAcquisition
{
	/// <summary>
	/// Description of LinearSensor.
	/// </summary>
	public class LinearSensor : Sensor
	{
		protected double intercept;
		protected double slope;
		
		
		public LinearSensor(string name, string quantity, string unit, double intercept, double slope) : base(name, quantity, unit)
		{
			this.intercept = intercept;
			this.slope = slope;
		}
		
		public LinearSensor(double intercept, double slope) : this(String.Empty, String.Empty, String.Empty, intercept, slope) { }
	
		public LinearSensor() : this(0.0, 1.0) { }
		
	
		public double Intercept {
			get { return intercept; }
			set { intercept = value; }
		}
		
		public double Slope {
			get { return slope; }
			set { slope = value; }
		}
		
		
		public override void Calibrate(IList<double> voltageValues, IList<double> quantityValues)
		{
			LinearFitResults fitResults = Statistics.GetLinearFit(voltageValues, quantityValues);
			
			intercept = fitResults.Intercept;
			slope = fitResults.Slope;
		}
		
		public override Sensor Clone()
		{
			return new LinearSensor(name, quantity, unit, intercept, slope);
		}
		
		public override void FromXml(XmlElement xmlElement)
		{
			base.FromXml(xmlElement);
			
			if (xmlElement.Attributes["type"].Value != "Linear") {
				Exception exception = new ArgumentException("Invalid sensor type.");
				
				exception.Data.Add("Expected type", "Linear");
				exception.Data.Add("Received type", xmlElement.Attributes["type"].Value);
				
				throw exception;
			}
			
			if (xmlElement.HasChildNodes) {
				XmlNode child;
				double parsed;
				
				child = xmlElement.SelectSingleNode("child::Intercept");
				intercept = Double.TryParse(child.InnerXml, out parsed) ? parsed : Double.NaN;
				
				child = xmlElement.SelectSingleNode("child::Slope");
				slope = Double.TryParse(child.InnerXml, out parsed) ? parsed : Double.NaN;
			}
		}
		
		public override string GetParameterString(params string[] format)
		{
			string interceptFormat = ((format != null) && (format.Length > 0)) ? format[0] : "g3";
			string slopeFormat = ((format != null) && (format.Length > 1)) ? format[1] : "g3";
			string parameterString = String.Format("Intercept: {0}", intercept.ToString(interceptFormat));
			
			if (!String.IsNullOrEmpty(unit)) {
				parameterString += String.Format(" {0}", unit);
			}
			
			parameterString += Environment.NewLine;
			parameterString += String.Format("Slope: {0}", slope.ToString(slopeFormat));
			parameterString += !String.IsNullOrEmpty(unit) ? String.Format(" ({0})/V", unit) : " 1/V";
			
			return parameterString;
		}
		
		public override double GetValue(double voltage)
		{
			return intercept + slope * voltage;
		}
	
		public override XmlElement ToXml(XmlDocument context)
		{
			XmlElement sensorXml = base.ToXml(context);
			XmlElement interceptXml = context.CreateElement("Intercept");
			XmlElement slopeXml = context.CreateElement("Slope");
		
			sensorXml.SetAttribute("type", "Linear");
			
			interceptXml.SetAttribute("unit", unit);
			interceptXml.InnerXml = intercept.ToString();
			sensorXml.AppendChild(interceptXml);
			
			slopeXml.SetAttribute("unit", !String.IsNullOrEmpty(unit) ? String.Format("({0})/V", unit) : "1/V");
			slopeXml.InnerXml = slope.ToString();
			sensorXml.AppendChild(slopeXml);
			
			return sensorXml;
		}
	}
}
