﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.01.21.
 * Idő: 9:04
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */

using System;
using System.Collections.Generic;
using System.Xml;
//
using TetheredSun.Core;

namespace Noise.DataAcquisition
{
	/// <summary>
	/// Description of Sensor.
	/// </summary>
	public abstract class Sensor : IXmlConvertible
	{
		protected string name;
		protected string quantity;
		protected string unit;
		
		protected Sensor(string name, string quantity, string unit)
		{
			// TODO: Implement automatic naming for null or empty strings.
			this.name = name;
			this.quantity = quantity;
			this.unit = unit;
		}
		
		protected Sensor() : this(String.Empty, String.Empty, String.Empty) { }
	
		
		public string Name {
			get { return name; }
			set { name = value; }
		}
		
		public string Quantity {
			get { return quantity; }
			set { quantity = value; }
		}
		
		public virtual string Unit {
			get { return unit; }
			set { unit = value; }
		}
		
		public virtual String ParameterString {
			get { return GetParameterString(null); }
		}
		
		public abstract void Calibrate(IList<double> voltageValues, IList<double> quantityValues);
		
		public abstract Sensor Clone();
		
		public virtual void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "Sensor") throw new ArgumentException("Cannot load Sensor data from a different xml element.");
			
			if (xmlElement.HasChildNodes) {
				XmlNode child;
				
				child = xmlElement.SelectSingleNode("child::Name");
				if (child != null) {
					name = child.InnerXml;
				}
				
				child = xmlElement.SelectSingleNode("child::Quantity");
				if (child != null) {
					quantity = child.InnerXml;
				}
				
				child = xmlElement.SelectSingleNode("child::Unit");
				if (child != null) {
					unit = child.InnerXml;
				}
			}
		}
		
		public string GetLabel()
		{
			if (String.IsNullOrEmpty(quantity)) {
				return "U [V]";
			} else {
				return quantity + (String.IsNullOrEmpty(unit) ? String.Empty :
				                   String.Format(" [{0}]", unit));
			}
		}
		
		public abstract string GetParameterString(params string[] format);
		
		public abstract double GetValue(double voltage);
		
		public virtual double[] GetValues(double[] voltages)
		{
			double[] values = new double[voltages.Length];
			
			for (int i = 0; i < values.Length; i++) {
				values[i] = GetValue(voltages[i]);
			}
			
			return values;
		}
		
		public string GetValueString(double value, string format, bool includeQuantity)
		{
			string valueString = String.Empty;
			
			if (includeQuantity) {
				valueString = String.Format("{0} = ", String.IsNullOrEmpty(quantity) ? "U" : quantity);
			}
			
			valueString += String.Format("{0} {1}", value.ToString(format), String.IsNullOrEmpty(unit) ? "V" : unit);
			
			return valueString;
		}
		
		public static Sensor Open(string path)
		{
			XmlDocument document = new XmlDocument();
			string type;
			Sensor sensor;
			
			document.Load(path);
			
			if (document.DocumentElement.Name != "Sensor") {
				Exception exception = new Exception("Invalid xml element name.");
				
				exception.Data.Add("Expected name", "Sensor");
				exception.Data.Add("Received name", document.DocumentElement.Name);
				
				throw exception;
			}
			
			type = document.DocumentElement.Attributes["type"].Value;
			
			switch (type) {
				case "Linear":
					sensor = new LinearSensor();
					break;
				case "Thermistor":
					sensor = new Thermistor();
					break;
				default:
					return null;
			}
			
			sensor.FromXml(document.DocumentElement);
			
			return sensor;
		}
		
		public virtual void Save(string path)
		{
			XmlDocument document = new XmlDocument();
			
			document.AppendChild(document.CreateXmlDeclaration("1.0", "utf-8", "yes"));
			document.AppendChild(this.ToXml(document));
			
			document.Save(path);
		}
		
		public virtual XmlElement ToXml(XmlDocument context)
		{
			XmlElement sensorXml = context.CreateElement("Sensor");
			XmlElement nameXml = context.CreateElement("Name");
			XmlElement quantityXml = context.CreateElement("Quantity");
			XmlElement unitXml = context.CreateElement("Unit");
			
			nameXml.InnerXml = name;
			sensorXml.AppendChild(nameXml);
			quantityXml.InnerXml = quantity;
			sensorXml.AppendChild(quantityXml);
			unitXml.InnerXml = unit;
			sensorXml.AppendChild(unitXml);
			
			return sensorXml;
		}
	}
}
