﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: Ouroboros
 * Dátum: 2010.01.17.
 * Idő: 14:04
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections.Generic;
using System.Xml;
//
using TetheredSun.Core.Mathematics;

namespace Noise.DataAcquisition
{
	public enum TemperatureScale {
		Celsius,
		Kelvin
	}
	
	/// <summary>
	/// Description of Thermistor.
	/// </summary>
	public class Thermistor : Sensor
	{
		public static double ZeroCelsius = 273.15;
		public static double DefaultCoefficient = 3977.0;
		public static double DefaultReferenceResistance = 10000.0;
		public static double DefaultProbeResistance = 10000.0;
		protected double maxVoltage = Edaq530.MaxVoltage;
		protected double coefficient;
		protected double referenceResistance;
		protected double referenceTemperature = ToKelvins(25);
		protected double probeResistance;
		protected double rInfinity;
		protected TemperatureScale temperatureScale = TemperatureScale.Celsius;
		
		
		
		
		public Thermistor(string name, string quantity, string unit, double coefficient,
		                  double referenceResistance, double probeResistance) : base(name, quantity, unit)
		{
			this.coefficient = coefficient;
			this.referenceResistance = referenceResistance;
			this.probeResistance = probeResistance;
			this.Unit = unit;
			CalculateRInfinity();
		}
		
		public Thermistor(string name, double coefficient, double referenceResistance,
		                  double probeResistance) : this(name, "T", "°C", coefficient, referenceResistance, probeResistance) { }
		
		
		public Thermistor(string name, double coefficient, double referenceResistance) :
			this(name, coefficient, referenceResistance, DefaultProbeResistance) { }
		
		public Thermistor(string name) : this(name, DefaultCoefficient, DefaultReferenceResistance, DefaultProbeResistance) { }
		
		public Thermistor() : this(String.Empty) { }
		
		
		
		public double MaxVoltage {
			get { return maxVoltage; }
		}
		
		public double Coefficient {
			get { return coefficient; }
			set { 
				coefficient = value;
				CalculateRInfinity();
			}
		}
		
		public double ReferenceResistance {
			get { return referenceResistance; }
			set { 
				referenceResistance = value; 
				CalculateRInfinity();
			}
		}
		
		public double ReferenceTemperature {
			get { return referenceTemperature; }
		}
		
		public double ProbeResistance {
			get { return probeResistance; }
			set { probeResistance = value; }
		}
		
		public TemperatureScale TemperatureScale {
			get { return temperatureScale; }
			set {
				temperatureScale = value;
				this.unit = (temperatureScale == TemperatureScale.Celsius) ? "°C" : "K";
			}
		}
		
		public override string Unit {
			get { return unit; }
			set {
				if (value.ToUpper().Trim() == "K") {
					TemperatureScale = TemperatureScale.Kelvin;
				} else {
					TemperatureScale = TemperatureScale.Celsius;
				}
			}
		}
		
		public override void Calibrate(IList<double> voltageValues, IList<double> quantityValues)
		{
			double[] lnResistances = new double[voltageValues.Count];
			double[] tReciprocals = new double[quantityValues.Count];
			double lnR0;
			LinearFitResults fitResults;
			
			// Thermistor formula: R(T) = R(infinity) * exp(B / T),
			// where R(infinity) := R(T0) * exp (-B / T0).
			// Linearisation yields: ln(R) = ln(R_infinity) + B * 1.0 / T,
			// which is a linear function, so linear fit can be applied.
			
			for (int i = 0; i < lnResistances.Length; i++) {
				lnResistances[i] = Math.Log(GetResistance(voltageValues[i]));
			}
			
			for (int i = 0; i < tReciprocals.Length; i++) {
				tReciprocals[i] = 1.0 / ((temperatureScale == TemperatureScale.Celsius) ?
				                         ToKelvins(quantityValues[i]) : quantityValues[i]);
			}
			
			fitResults = Statistics.GetLinearFit(tReciprocals, lnResistances);
			
			// The slope is B, the intercept is ln(R_infinity).
			coefficient = fitResults.Slope;
			// Fix reference temperature:
			referenceTemperature = ToKelvins(25.0);
			
			// ln(R_infinity) = ln(R_0) - B / T0.
			// The intercept of the fit is ln(R_infinity).
			lnR0 = fitResults.Intercept + coefficient / referenceTemperature;
			referenceResistance = Math.Exp(lnR0);
			CalculateRInfinity();
		}
		
		public override Sensor Clone()
		{
			Sensor clone = new Thermistor(name, quantity, unit, coefficient, referenceResistance, probeResistance);
			
			return clone;
		}
		
		public override void FromXml(XmlElement xmlElement)
		{
			base.FromXml(xmlElement);
			
			if (xmlElement.Attributes["type"].Value != "Thermistor") {
				Exception exception = new ArgumentException("Invalid sensor type.");
				
				exception.Data.Add("Expected type", "Thermistor");
				exception.Data.Add("Received type", xmlElement.Attributes["type"].Value);
				
				throw exception;
			}
			
			switch (this.unit) {
				case "K":
					temperatureScale = TemperatureScale.Kelvin;
					break;
				case "°C":
					temperatureScale = TemperatureScale.Celsius;
					break;
				default:
					this.unit = "°C";
					temperatureScale = TemperatureScale.Celsius;
					break;
			}
			
			if (xmlElement.HasChildNodes) {
				XmlNode child;
				double parsed;
				
				child = xmlElement.SelectSingleNode("child::Coefficient");
				coefficient = Double.TryParse(child.InnerXml, out parsed) ? parsed : DefaultCoefficient;
				
				// TODO: Implement custom reference temperatures.
				
				child = xmlElement.SelectSingleNode("child::Reference/Resistance");
				referenceResistance = Double.TryParse(child.InnerXml, out parsed) ? parsed : DefaultReferenceResistance;
				
				child = xmlElement.SelectSingleNode("child::ProbeResistance");
				probeResistance = Double.TryParse(child.InnerXml, out parsed) ? parsed : DefaultProbeResistance;
				
				CalculateRInfinity();
			}
		}
		
		public override string GetParameterString(params string[] format)
		{
			string r0Format = ((format != null) && (format.Length > 0)) ? format[0] : "f0";
			string t0Format = ((format != null) && (format.Length > 1)) ? format[1] : "f2";
			string coefficientFormat = ((format != null) && (format.Length > 2)) ? format[2] : "f0";
			string probeFormat = ((format != null) && (format.Length > 3)) ? format[3] : "f0";
			string parameterString;
			
			parameterString = String.Format("Reference: {0} Ω at {1} K",
			                                referenceResistance.ToString(r0Format),
			                                referenceTemperature.ToString(t0Format));
			parameterString += Environment.NewLine;
			parameterString += String.Format("Coefficient: {0} K", coefficient.ToString(coefficientFormat));
			parameterString += Environment.NewLine;
			parameterString += String.Format("Probe resistance: {0} Ω", probeResistance.ToString(probeFormat));
			
			return parameterString;
		}
		
		public override double GetValue(double voltage)
		{
			double resistance = GetResistance(voltage);
			double t =  coefficient / Math.Log(resistance / rInfinity);
			
			return (temperatureScale == TemperatureScale.Celsius) ? ToCelsius(t) : t;
		}
		
		public static double ToCelsius(double kelvin)
		{
			return kelvin - ZeroCelsius;
		}
		
		public static double ToKelvins(double centigrade)
		{
			return centigrade + ZeroCelsius;
		}
		
		public override XmlElement ToXml(XmlDocument context)
		{
			XmlElement thermistorXml = base.ToXml(context);
			XmlElement coefficientXml = context.CreateElement("Coefficient");
			XmlElement referenceXml = context.CreateElement("Reference");
			XmlElement referenceTemperatureXml = context.CreateElement("Temperature");
			XmlElement referenceResistanceXml = context.CreateElement("Resistance");
			XmlElement probeResistanceXml = context.CreateElement("ProbeResistance");
			
			thermistorXml.SetAttribute("type", "Thermistor");
			
			coefficientXml.SetAttribute("unit", "K");
			coefficientXml.InnerXml = coefficient.ToString();
			thermistorXml.AppendChild(coefficientXml);
			
			referenceTemperatureXml.SetAttribute("unit", "K");
			referenceTemperatureXml.InnerXml = referenceTemperature.ToString();
			referenceXml.AppendChild(referenceTemperatureXml);
			
			referenceResistanceXml.SetAttribute("unit", "Ω");
			referenceResistanceXml.InnerXml = referenceResistance.ToString();
			referenceXml.AppendChild(referenceResistanceXml);
			thermistorXml.AppendChild(referenceXml);
			
			probeResistanceXml.SetAttribute("unit", "Ω");
			probeResistanceXml.InnerXml = probeResistance.ToString();
			thermistorXml.AppendChild(probeResistanceXml);
			
			return thermistorXml;
		}
		
		protected double GetCoefficient(double resistance1, double temperature1, double resistance2, double temperature2)
		{
			return (Math.Log(resistance1) - Math.Log(resistance2)) *
				(temperature1 * temperature2 / (temperature2 - temperature1));
		}
		
		protected void CalculateRInfinity()
		{
			rInfinity = referenceResistance * Math.Exp(-coefficient / referenceTemperature);
		}
		
		protected double GetResistance(double voltage)
		{
			return probeResistance * voltage / (maxVoltage - voltage);
		}
	}
}
