﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: Ouroboros
 * Dátum: 2009.06.14.
 * Idő: 11:05
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Drawing;
using TetheredSun.Core.Mathematics;

namespace TetheredSun.Core.GnuPlot
{
	/// <summary>
	/// An axis for GnuPlot scripts.
	/// </summary>
	public class Axis
	{
		protected string label = String.Empty;
		protected string format = "%.0f";
		protected FontInfo labelFont = new FontInfo("Helvetica", 16.0f);
		protected FontInfo ticsFont = new FontInfo("Helvetica", 12.0f);
		protected bool majorGrid = true;
		protected bool minorGrid = false;
		protected bool ticsMirrored = false;
		protected bool rotateLabel = false;
		protected double[] ticsOffset = null;
		protected double[] labelOffset = null;
		protected Interval range = null;
		protected int? logBase = null;
		
		
		public Axis(string label)
		{
			this.label = label;
		}
		
		public Axis() { }
		
		
		public string Label {
			get { return label; }
			set { label = value; }
		}
		
		public double[] LabelOffset {
			get { return labelOffset; }
			set { labelOffset = value; }
		}
				
		public string Format {
			get { return format; }
			set { format = value; }
		}
		
		public FontInfo LabelFont {
			get { return labelFont; }
			set { labelFont = value; }
		}
		
		public FontInfo TicsFont {
			get { return ticsFont; }
			set { ticsFont = value; }
		}
		
		public bool TicsMirrored {
			get { return ticsMirrored; }
			set { ticsMirrored = value; }
		}
				
		public double[] TicsOffset {
			get { return ticsOffset; }
			set { ticsOffset = value; }
		}
		
		public bool MajorGrid {
			get { return majorGrid; }
			set { majorGrid = value; }
		}
		
		public bool MinorGrid {
			get { return minorGrid; }
			set { minorGrid = value; }
		}
		
		public Interval Range {
			get { return range; }
			set { range = value; }
		}
		
		public bool RotateLabel {
			get { return rotateLabel; }
			set { rotateLabel = value; }
		}
				
		public Nullable<int> LogBase {
			get { return logBase; }
			set { logBase = value; }
		}
		
		public string ToGnuPlotString(AxisType type)
		{
			string text;
			string axisString;
			
			switch (type) {
				case AxisType.Y:
					axisString = "y";
					break;
				case AxisType.Z:
					axisString = "z";
					break;
				case AxisType.X:
				default:
					axisString = "x";
					break;
			}
			
			text = String.Format("set {0}tics {1}mirror", axisString, ticsMirrored ? String.Empty : "no");
			if ((ticsOffset != null) && (ticsOffset.Length > 1)) {
				text += String.Format(" offset {0}, {1}", ticsOffset[0].ToString("f1").Replace(',', '.'), ticsOffset[1].ToString("f1").Replace(',', '.'));
			}
			if (ticsFont != null) {
				text += String.Format(" font \"{0}, {1}\"", ticsFont.Name, ticsFont.Size);
			}
			text += Environment.NewLine;
			if (majorGrid || minorGrid) {
				text += "set grid" + (minorGrid ? " m" : " ") + axisString + "tics" + Environment.NewLine;
			}
			if (!String.IsNullOrEmpty(format)) {
				text += "set format " + axisString + " \"" + format + "\"" + Environment.NewLine;
			}
			if (!String.IsNullOrEmpty(label)) {
				text += String.Format("set {0}label \"{1}\"", axisString, label);
				if ((labelOffset != null) && (labelOffset.Length > 1)) {
					text += String.Format(" offset {0}, {1}", labelOffset[0].ToString("f1").Replace(',', '.'), labelOffset[1].ToString("f1").Replace(',', '.'));
				}
				if (labelFont != null) {
					text += String.Format(" font \"{0}, {1}\"", labelFont.Name, labelFont.Size);
				}
				if (rotateLabel) text += " rotate";
				text += Environment.NewLine;
			}
			if (range != null) {
				string min = Double.IsNegativeInfinity(range.Minimum) ? String.Empty : range.Minimum.ToString();
				string max = Double.IsPositiveInfinity(range.Maximum) ? String.Empty : range.Maximum.ToString();
				text += String.Format("set {0}range [{1}:{2}]", axisString, min, max) + Environment.NewLine;
			}
			if (logBase.HasValue) {
				text += String.Format("set logscale {0} {1}", axisString, logBase.Value) + Environment.NewLine;
			}
			
			return text;
		}
	}
}
