﻿/*
 * Created by SharpDevelop.
 * User: Ouroboros
 * Date: 2006.11.04.
 * Time: 14:55
 * Version: 0.3.0
 * Latest update: property calls replaced with field calls to improve performance.
 * Updates:
 * To change this template use Tools | Options | Coding | Edit Standard Headers.
 */
using System;
using System.Collections;
using System.Collections.Generic;

namespace TetheredSun.Core.Mathematics
{
	/// <summary>
	/// A class representing complex numbers.
	/// </summary>
	public class Complex : IEquatable<Complex>
	{
		protected double re = 0.0;	// Real part.
		protected double im = 0.0;	// Imaginary part.
	
		
		public Complex(double re, double im)
		{
			this.re = re;
			this.im = im;
		}
		
		public Complex() : this(0.0, 0.0) { }
		
		
		
		public static readonly Complex Zero = new Complex(0.0D, 0.0D);
		public static readonly Complex NaN = new Complex(Double.NaN, Double.NaN);
				
			
		public static Complex operator +(Complex c1, Complex c2)
		{
			return new Complex(c1.re + c2.re, c1.im + c2.im);
		}
		
		public static Complex operator +(Complex c, double d)
		{
			return new Complex(c.re + d, c.im);
		}
		
		public static Complex operator +(double d, Complex c)
		{
			return new Complex(c.re + d, c.im);
		}
		
		public static Complex operator -(Complex c1, Complex c2)
		{
			return new Complex(c1.re - c2.re, c1.im - c2.im);
		}
		
		public static Complex operator -(Complex c, double d)
		{
			return new Complex(c.re - d, c.im);
		}
		
		public static Complex operator -(double d, Complex c)
		{
			return new Complex(d - c.re, -c.im);
		}
		
		public static Complex operator *(Complex c1, Complex c2)
		{
			return new Complex(c1.re * c2.re - c1.im * c2.im, c1.im * c2.re + c1.re * c2.im);
		}
		
		public static Complex operator *(double d, Complex c)
		{
			return new Complex(d * c.re, d * c.im);
		}
		
		public static Complex operator *(Complex c, double d)
		{
			return new Complex(d * c.re, d * c.im);
		}
		
		public static Complex operator /(Complex c1, Complex c2)
		{
			if ((c2.re != 0.0D) || (c2.im != 0.0D)) {
				double abs = c1.Abs() / c2.Abs();
				double arg = c1.Arg() - c2.Arg();
					
				return ToReIm(abs, arg);
			} else {
				return NaN;
			}
			
		}
		
		public static Complex operator /(Complex c, double d)
		{
			if (d != 0.0) {
				return new Complex(c.re / d, c.im / d);
			} else {
				return Complex.NaN;
			}
		}
		
		public static Complex operator /(double d, Complex c)
		{
			Complex cp = new Complex(d, 0.0);
			
			return (c != Complex.Zero) ? cp / c : Complex.NaN;
		}
		
		public static bool operator ==(Complex c1, Complex c2)
		{
			return ((c1.re == c2.re) && (c1.im == c2.im));
		}
		
		public static bool operator !=(Complex c1, Complex c2)
		{
			return !((c1.re == c2.re) && (c1.im == c2.im));
		}
				
		public static implicit operator Complex(double d)
		{
			return new Complex(d, 0.0);
		}
		
		public static Complex ToReIm(double magnitude, double phaseInRadians)
		{
			return new Complex(magnitude * Math.Cos(phaseInRadians), magnitude * Math.Sin(phaseInRadians));
		}
		
		public static Complex ToReImDeg(double magnitude, double phaseInDegrees)
		{
			double phaseInRadians = (phaseInDegrees / 180.0D) * Math.PI;
			
			return new Complex(magnitude * Math.Cos(phaseInRadians), magnitude * Math.Sin(phaseInRadians));
		}
		
		public static Complex Exp(double x)	// Implement the exp(i*x) function;
		{
			return new Complex(Math.Cos(x), Math.Sin(x));
		}
		
		// TODO: Test it.
		public static bool TryParse(string s, out Complex result)
		{
			bool success;
			double r;
			double i;
			string[] split;
			string bare = s.Replace(" ", String.Empty);
			
			if (bare.Contains("+i*")) {
				split = bare.Split(new string[] { "+i*" }, StringSplitOptions.RemoveEmptyEntries);
				success = Double.TryParse(split[0], out r);
				success &= Double.TryParse(split[1], out i);
				if (success) {
					result = new Complex(r, i);
				} else {
					result = null;
				}
			} else if (bare.Contains("-i*")) {
				split = bare.Split(new string[] { "-i*" }, StringSplitOptions.RemoveEmptyEntries);
				if (split.Length > 1) {
					success = Double.TryParse(split[0], out r);
					success &= Double.TryParse(split[1], out i);
					if (success) {
						result = new Complex(r, -i);
					} else {
						result = null;
					}
				} else {
					success = Double.TryParse(split[0], out i);
					if (success) {
						result = new Complex(0.0, -i);
					} else {
						result = null;
					}
				}
			} else if (bare.Contains("i*")) {
				split = bare.Split(new string[] { "i*" }, StringSplitOptions.RemoveEmptyEntries);
				success = Double.TryParse(split[0], out i);
				if (success) {
					result = new Complex(0.0, i);
				} else {
					result = null;
				}
			} else {
				success = Double.TryParse(bare, out r);
				if (success) {
					result = new Complex(r, 0.0);
				} else {
					result = null;
				}
			}
			
			return success;
		}
		
		public string ToString(string format)
		{
			string str = String.Empty;
			bool formatted = !String.IsNullOrEmpty(format);
			
			if (Math.Abs(re) > 0) {	// Real part not 0;
				str += formatted ? re.ToString(format) : re.ToString();
				if (Math.Abs(im) > 0) {	// Imaginary part not 0;
					if(im > 0)
						str += " + i * " + (formatted ? im.ToString(format) : im.ToString());
					else
						str += " - i * " + (formatted ? (Math.Abs(im)).ToString(format) : (Math.Abs(im)).ToString());
				}
			} else {	// Real part 0;
				if (Math.Abs(im) > 0) {	// Imaginary part not 0;
					if(im > 0)
						str += "i * " + (formatted ? im.ToString(format) : im.ToString());
					else
						str += "-i * " + (formatted ? (Math.Abs(im)).ToString(format) : (Math.Abs(im)).ToString());
				} else {
					// Whole complex number zero;
					str += "0";
				}
					
			}
			
			return str;
		}
		
		public override string ToString()
		{
			return ToString(null);
		}
		
		public double Arg()
		{
			if ((re != 0) || (im != 0)) {
				double result = Math.Atan2(im, re);
				return result;
			} else {
				return Double.NaN;
			}
		}
		
		public double Abs()
		{
			return Math.Sqrt(re * re + im * im);
		}
		
		public double AbsoluteSquare()
		{
			return re * re + im * im;
		}
		
		public Complex Conjugate()
		{
			return new Complex(re, -im);
		}
		
		public override bool Equals(object c)
		{
			if (c == null || GetType() != c.GetType()) return false;
			Complex cp = (Complex)c;
			
			return (re == cp.re) && (im == cp.im);
		}
		
		public bool Equals(Complex other)
		{
			return (re == other.re) && (im == other.im);
		}
		
		public override int GetHashCode()
		{
			return (int)re ^ (int)im;	// Provide a unique hash code for identification.
		}
		
		
		public double Re {
			get {
				return re;
			}
			set {
				re = value;
			}
		}

		public double Im {
			get {
				return im;
			}
			set {
				im = value;
			}
		}
	
		public bool IsNaN {
			get { return Double.IsNaN(re) && Double.IsNaN(im); }
		}
	}
}
