﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: Ouroboros
 * Dátum: 2009.06.14.
 * Idő: 9:11
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */
using System;
using System.Collections.Generic;
using System.IO;
using System.Drawing;
using TetheredSun.Core.Mathematics;

namespace TetheredSun.Core.GnuPlot
{
	public enum AxisType {
		X,
		Y,
		Z
	}
	
	public enum Terminal {
		EpsMonochrome,
		EpsColour
	}
	
	[Flags]
	public enum Border {		// In 'plot' | in 'splot' 
		Bottom = 1,				// bottom | bottom left front
		Left = 2,				// left | bottom left back
		Top = 4,				// top | bottom right front
		Right = 8,				// right | bottom right back
		LeftVertical = 16,		// — | left vertical
		BackVertical = 32,		// — | back vertical
		RightVertical = 64,		// — | right vertical
		FrontVertical = 128,	// — | front vertical
		TopLeftBack = 256,		// — | top left back
		TopRightBack = 512,		// — | top right back
		TopLeftFront = 1024, 	// — | top left front
		TopRightFront = 2048	// — | top right front
	}
	
	/// <summary>
	/// Description of GnuPlotTextWriter.
	/// </summary>
	public class GnuPlotTextWriter : StreamWriter
	{
		protected Axis axisX = new Axis("x");
		protected Axis axisY = new Axis("y");
		protected Axis axisZ = new Axis("z");
		protected FontInfo baseFont = new FontInfo("Helvetica", 12.0f);
		protected string outputPath = "Plot.eps";
		protected Terminal terminal;
		
		
		public GnuPlotTextWriter(Stream stream, Terminal terminal) : base(stream)
		{ 
			this.terminal = terminal;
		}
		
		public GnuPlotTextWriter(string path, Terminal terminal) : base(path)
		{
			this.terminal = terminal;
		}
		
		public GnuPlotTextWriter(string path) : this(path, Terminal.EpsMonochrome) { }
		
		
		public Axis AxisX {
			get { return axisX; }
			set { axisX = value; }
		}
		
		public Axis AxisY {
			get { return axisY; }
			set { axisY = value; }
		}
		
		public Axis AxisZ {
			get { return axisZ; }
			set { axisZ = value; }
		}
		
		public FontInfo BaseFont {
			get { return baseFont; }
			set { baseFont = value; }
		}
		
		public string OutputPath {
			get { return outputPath; }
			set { outputPath = value; }
		}
		
		public Terminal Terminal {
			get { return terminal; }
			set { terminal = value; }
		}
		
		
		public string Quote(string text)
		{
			return String.IsNullOrEmpty(text) ? text : "\"" + text + "\"";
		}
		
		public void WriteAxisX()
		{
			if (axisX != null) this.Write(axisX.ToGnuPlotString(AxisType.X));
		}
		
		public void WriteAxisY()
		{
			if (axisY != null) this.Write(axisY.ToGnuPlotString(AxisType.Y));
		}
		
		public void WriteAxisZ()
		{
			if (axisZ != null) this.Write(axisZ.ToGnuPlotString(AxisType.Z));
		}
		
		public void WriteBorder(Border border)
		{
			this.WriteLine("set border " + ((int)border).ToString());
		}
		
		public void WriteComment(string comment)
		{
			this.WriteLine("# " + comment);
		}
		
		public void WriteInlineData(double[][] data, string format, string separator)
		{
			if (data == null) return;
			string line;
			
			for (int i = 0; i < data.Length; i++) {
				if ((data[i] == null) || (data[i].Length <= 0)) continue;
				line = data[i][0].ToString(format).Replace(',', '.');
				for (int j = 1; j < data[i].Length; j++) {
					line += separator + data[i][j].ToString(format).Replace(',', '.');
				}
				this.WriteLine(line);
			}
			this.WriteLine();
			this.WriteLine("e");
		}
		
		public void WriteInlineData(double[][] data)
		{
			WriteInlineData(data, "f3", "\t");
		}
		
		public void WriteInlineData(Complex[][] data, string format, string separator)
		{
			if (data == null) return;
			double[][] absoluteValue = new double[data.Length][];
			
			for (int i = 0; i < absoluteValue.Length; i++) {
				absoluteValue[i] = new double[data[i].Length];
				for (int j = 0; j < data[i].Length; j++) {
					absoluteValue[i][j] = data[i][j].Abs();
				}
			}
			
			WriteInlineData(absoluteValue, format, separator);
		}
		
		public void WriteInlineData(Complex[][] data)
		{
			WriteInlineData(data, "f3", "\t");
		}
		
		public void WriteXYData(double[] x, double[][] y, string format, string separator)
		{
			if (x == null) return;
			if (y == null) return;
			string line;
			
			for (int i = 0; i < x.Length; i++) {
				line = x[i].ToString(format).Replace(',', '.');
				for (int j = 0; j < y.Length; j++) {
					line += separator + y[j][i].ToString(format).Replace(',', '.');
				}
				this.WriteLine(line);
			}
			this.WriteLine("e");
		}
		
		public void WriteXYData(double[] x, double[][] y)
		{
			WriteXYData(x, y, "f3", "\t");
		}
		
		public void WriteOutput(bool includePath)
		{
			this.Write("set output");
			if (includePath && !String.IsNullOrEmpty(outputPath)) {
				this.WriteLine(" " + Quote(outputPath));
			} else {
				this.WriteLine();
			}
		}
		
		public void WriteSeparatorLine(int length)
		{
			string text = String.Empty;
			
			for (int i = 0; i < length; i++) {
				text += "#";
			}
			
			this.WriteLine(text);
		}
				
		public void WriteTerminal()
		{
			string text = "set terminal";
			
			switch (terminal) {
				case Terminal.EpsColour:
					text += " postscript eps enhanced colour";
					break;
				case Terminal.EpsMonochrome:
				default:
					text += " postscript eps enhanced monochrome";
					break;
			}
			text += String.Format(" {0} {1}", Quote(baseFont.Name), baseFont.Size);
			this.WriteLine(text);
		}
	
		public void WriteVariable(string name, string value)
		{
			if (String.IsNullOrEmpty(name)) return;
			
			this.WriteLine(String.Format("{0} = {1}", name, value));
		}
		
		public void WriteVariable(string name, double value)
		{
			WriteVariable(name, value.ToString().Replace(',', '.'));
		}
	
		public static string GetColourString(Color colour)
		{
			string colourString = "#";
			
			colourString += colour.R.ToString("X2");
			colourString += colour.G.ToString("X2");
			colourString += colour.B.ToString("X2");
			
			return colourString;
		}
	}
}
