﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.06.08.
 * Idő: 12:57
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */

using System;
using System.Xml;

namespace TetheredSun.Core
{
	/// <summary>
	/// Description of Interval.
	/// </summary>
	public class Interval : IEquatable<Interval>, IXmlConvertible
	{
		protected double minimum;
		protected double maximum;
		protected double sum;
		protected bool includeMinimum;
		protected bool includeMaximum;
		protected string name = String.Empty;
		protected string unit = String.Empty;
		
		public Interval(double minimum, double maximum, bool includeMinimum, bool includeMaximum)
		{
			this.minimum = minimum;
			this.maximum = maximum;
			this.includeMinimum = includeMinimum;
			this.includeMaximum = includeMaximum;
			this.sum = 0.0;
		}
		
		public Interval(double minimum, double maximum) : this(minimum, maximum, true, true) { }
		
		public Interval() : this(Double.NaN, Double.NaN, false, false) { }
		
		public bool IncludeMinimum {
			get { return includeMinimum; }
		}
		
		public bool IncludeMaximum {
			get { return includeMaximum; }
		}
				
		public double Minimum {
			get { return minimum; }
		}
		
		public double Maximum {
			get { return maximum; }
		}
		
		public string Name {
			get { return name; }
			set { name = value; }
		}
		
		public double Sum {
			get { return sum; }
		}
		
		public string Unit {
			get { return unit; }
			set { unit = value; }
		}
		
		public double Width {
			get { return maximum - minimum; }
		}
				
		
		public bool Includes(double value)
		{
			bool includes = (includeMinimum ? (value >= minimum) : (value > minimum));
			includes &= (includeMaximum ? (value <= maximum) : (value < maximum));
			
			return includes;
		}
		
		public bool Test(double x, double y)
		{
			bool includes = Includes(x);
			
			if (includes) sum += y;
			
			return includes;
		}
	
		/// <summary>
		/// Returns a <code>string</code> that represents the current <code>Interval.</code>
		/// </summary>
		/// <param name="format">A string specifying how the interval is formatted.
		/// The first character specifies the endpoint notation: [ for ISO, ( for American and N for none.
		/// The second character is the separator character.
		/// The third character specifies where to indicate the unit: 0 for none, 1 for the minimum only, 2 for the maximum only and 3 for both.
		/// </param>
		/// <returns>A <code>string</code> that represents the current <code>Interval.</code></returns>
		public string ToString(string format)
		{
			string startBracket;
			string endBracket;
			string separator;
			string startUnit;
			string endUnit;
			
			switch (format[0]) {
				case '[':
					startBracket = includeMinimum ? "[" : "]";
					endBracket =  includeMaximum ? "]" : "[";
					break;
				case '(':
					startBracket = includeMinimum ? "[" : "(";
					endBracket =  includeMaximum ? "]" : ")";
					break;
				case 'N':
				default:
					startBracket = endBracket = String.Empty;
					break;
			}
			
			separator = format[1].ToString();
			if ((separator == ",") || (separator == ";")) separator += " ";
			
			switch (format[2]) {
				case '1':
					startUnit = String.IsNullOrEmpty(unit) ? String.Empty : " " + unit;
					endUnit = String.Empty;
					break;
				case '2':
					startUnit = String.Empty;
					endUnit = String.IsNullOrEmpty(unit) ? String.Empty : " " + unit;
					break;
				case '3':
					startUnit = String.IsNullOrEmpty(unit) ? String.Empty : " " + unit;
					endUnit = String.IsNullOrEmpty(unit) ? String.Empty : " " + unit;
					break;
				case '0':
				default:
					startUnit = String.Empty;
					endUnit = String.Empty;
					break;
			}
			
			return startBracket + minimum.ToString() + startUnit + separator + maximum.ToString() + endUnit + endBracket;
		}
		
		
		public override string ToString()
		{
			return ToString("N—2");
		}
		
		
		public bool Equals(Interval other)
		{
			bool equals = true;
			
			equals &= (name == other.name);
			equals &= (unit == other.unit);
			equals &= (minimum == other.minimum);
			equals &= (maximum == other.maximum);
			equals &= (includeMinimum == other.includeMinimum);
			equals &= (includeMaximum == other.includeMaximum);
			
			return equals;
		}
		
		public void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "Interval") throw new ArgumentException("Cannot load Interval data from a different xml element.");
			if (xmlElement.HasChildNodes) {
				double minimumLocal = Double.NaN;
				double maximumLocal = Double.NaN;
				bool includeMinLocal = false;
				bool includeMaxLocal = false;
				bool success = true;
				string nameLocal = xmlElement.Attributes["name"].Value;
				string unitLocal = xmlElement.Attributes["unit"].Value;
				foreach (XmlNode child in xmlElement.ChildNodes) {
					if (!(child is XmlElement)) continue;
					switch (child.Name) {
						case "Minimum":
							success &= Double.TryParse(child.InnerXml, out minimumLocal);
							success &= Boolean.TryParse(child.Attributes["included"].Value, out includeMinLocal);
							break;
						case "Maximum":
							success &= Double.TryParse(child.InnerXml, out maximumLocal);
							success &= Boolean.TryParse(child.Attributes["included"].Value, out includeMaxLocal);
							break;
						default:
							break;
					}
				}
				
				if (success) {
					minimum = minimumLocal;
					includeMinimum = includeMinLocal;
					maximum = maximumLocal;
					includeMaximum = includeMaxLocal;
					name = nameLocal;
					unit = unitLocal;
				} else {
					minimum = Double.NaN;
					includeMinimum = false;
					maximum = Double.NaN;
					includeMaximum = false;;
					name = String.Empty;
					unit = String.Empty;
				}
			}
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement xml = context.CreateElement("Interval");
			XmlElement child;
			
			if (!String.IsNullOrEmpty(name)) {
				xml.SetAttribute("name", name);
			}
			
			if (!String.IsNullOrEmpty(unit)) {
				xml.SetAttribute("unit", unit);
			}
			
			child = context.CreateElement("Minimum");
			child.SetAttribute("included", includeMinimum.ToString());
			child.InnerXml = minimum.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("Maximum");
			child.SetAttribute("included", includeMaximum.ToString());
			child.InnerXml = maximum.ToString();
			xml.AppendChild(child);
			
			return xml;
		}
		
	}
}
