/*
 * Created by SharpDevelop.
 * User: phil
 * Date: 2010.02.19.
 * Time: 15:42
 * 
 * To change this template use Tools | Options | Coding | Edit Standard Headers.
 */
using System;
using System.Collections.Generic;
using System.Xml;
//
using TetheredSun.Core;

namespace TetheredSun.SignalAnalysis
{
	/// <summary>
	/// Description of LevelCrossingDetector.
	/// </summary>
	public class LevelCrossingDetector : IXmlConvertible
	{
		protected object syncRoot = new Object();
		protected double hysteresis = 0.0;
		protected double level;
		protected double oldTime = 0.0;				// The time of a previous crossing candidate.
		protected double oldValue = Double.NaN;		// The value at a previous crossing candidate.
		protected double lastTime = Double.NaN;		// The previous time instance, regardless of crossing.
		protected double lastValue = Double.NaN;	// The previous value tested, regardless of crossing.
		protected volatile bool isActive = true;
		protected bool downBegun = false;
		protected bool upBegun = false;
		
		public event EventHandler CyclesUpdated;
		
		
		public LevelCrossingDetector(double level, double hysteresis)
		{
			this.level = level;
			this.hysteresis = hysteresis;
		}
		
		public LevelCrossingDetector(double level) : this(level, 0.0) { }
		
		
		public double Hysteresis {
			get { return hysteresis; }
			set { hysteresis = value; }
		}
		
		public bool IsActive {
			get { return isActive; }
			set { isActive = value; }
		}
				
		public double Level {
			get { return level; }
			set { level = value; }
		}
	
		
		public void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "LevelCrossingDetector") {
				Exception exception = new ArgumentException("Invalid element name.");
				
				exception.Data.Add("Expected name", "LevelCrossingDetector");
				exception.Data.Add("Received name", xmlElement.Name);
				
				throw exception;
			}
			
			XmlNode child;
			double parsedDouble;
			bool parsedBool;
			
			child = xmlElement.SelectSingleNode("child::Level");
			level = Double.TryParse(child.InnerXml, out parsedDouble) ? parsedDouble : 1.0;
			
			child = xmlElement.SelectSingleNode("child::Hysteresis");
			hysteresis = Double.TryParse(child.InnerXml, out parsedDouble) ? parsedDouble : 0.0;
			
			child = xmlElement.SelectSingleNode("child::IsActive");
			isActive = Boolean.TryParse(child.InnerXml, out parsedBool) ? parsedBool : true;
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement detectorXml = context.CreateElement("LevelCrossingDetector");
			XmlElement levelXml = context.CreateElement("Level");
			XmlElement hysteresisXml = context.CreateElement("Hysteresis");
			XmlElement isActiveXml = context.CreateElement("IsActive");
			
			levelXml.InnerXml = level.ToString();
			detectorXml.AppendChild(levelXml);
			
			hysteresisXml.InnerXml = hysteresis.ToString();
			detectorXml.AppendChild(hysteresisXml);
			
			isActiveXml.InnerXml = isActive.ToString();
			detectorXml.AppendChild(isActiveXml);
			
			return detectorXml;
		}
		
		
		public bool Test(double time, double value, out LevelCrossing levelCrossing)
		{
			if (!isActive) {
				levelCrossing = null;
				return false;
			}
			if (Double.IsNaN(lastTime) || Double.IsNaN(lastValue)) {
				lastTime = time;
				lastValue = value;
				levelCrossing = null;
				return false;
			}
			
			bool crossingFound = false;
			double level1;
			double level2;
			
		
			level1 = level - hysteresis;
			level2 = level + hysteresis;
			
			levelCrossing = null;
			
			if (value > level2) {
				if (lastValue <= level1) {	// We have an immediate upwards crossing.
					// Clear crossing start indicators to be sure.
					upBegun = false;
					downBegun = false;
					oldTime = lastTime;
					oldValue = lastValue;
					crossingFound = true;
					levelCrossing = GetCrossing(time, value, LevelCrossingDirection.Up);
				} else if (lastValue <= level2) {
					if (upBegun) {		// Complete the upwards level-crossing candidate.
						upBegun = false;
						downBegun = false;
						crossingFound = true;
						levelCrossing = GetCrossing(time, value, LevelCrossingDirection.Up);
					} else if (downBegun) {	// Cancel the downwards level-crossing candidate.
						downBegun = false;
						crossingFound = false;
						levelCrossing = null;
					}
				}
			} else if ((value < level2) && (value > level1)) {
				if (lastValue >= level2) {	// We have a downwards level-crossing candidate.
					upBegun = false;
					downBegun = true;
					oldTime = time;
					oldValue = value;
					crossingFound = false;
					levelCrossing = null;
				} else if (lastValue <= level1) {
					upBegun = true;
					downBegun = false;
					oldTime = time;
					oldValue = value;
					crossingFound = false;
					levelCrossing = null;
				}
			} else if (value < level1) {
				if (lastValue >= level2) {	// We have an immediate downwards crossing.
					// Clear crossing start indicators to be sure.
					upBegun = false;
					downBegun = false;
					oldTime = lastTime;
					oldValue = lastValue;
					crossingFound = true;
					levelCrossing = GetCrossing(time, value, LevelCrossingDirection.Down);
				} else if (lastValue >= level1) {
					if (downBegun) {
						upBegun = false;
						downBegun = false;
						crossingFound = true;
						levelCrossing = GetCrossing(time, value, LevelCrossingDirection.Down);
					} else if (upBegun) {
						upBegun = false;
						crossingFound = false;
						levelCrossing = null;
					}
				}
			}
			
			lastTime = time;
			lastValue = value;
			return crossingFound;
		}
		
		protected LevelCrossing GetCrossing(double time, double value, LevelCrossingDirection direction)
		{
			double slope = (value - oldValue) / (time - oldTime);
			double t = oldTime + (level - oldValue) / slope;
			double x = oldValue + (t - oldTime) * slope;
			LevelCrossing crossing = new LevelCrossing(t, x, direction);
			
			return crossing;
		}
		
		protected virtual void OnCyclesUpdated(EventArgs e)
		{
			if (CyclesUpdated != null) {
				CyclesUpdated(this, e);
			}
		}
		
	}
}
