﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.05.07.
 * Idő: 15:43
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */

using System;
using System.Xml;

namespace TetheredSun.SignalAnalysis
{
	/// <summary>
	/// Description of ScalingByBounds.
	/// </summary>
	public class ScaleByBounds : IScale
	{
		protected int count;
		protected string name = String.Empty;
		protected string unit = String.Empty;
		protected double startValue;
		protected double endValue;
		protected double increment;	// May be a difference or a ratio, depending whether the scaling is logarithmic.
		protected bool isLogarithmic;
		
		public ScaleByBounds(int count, double startValue, double endValue, bool isLogarithmic)
		{
			this.count = count;
			this.startValue = startValue;
			this.endValue = endValue;
			this.isLogarithmic = isLogarithmic;
			increment  = isLogarithmic ? Math.Pow(endValue / startValue, 1.0 / (count - 1)) : (endValue - startValue) / (count - 1);
		}
		
		public ScaleByBounds(int count, double startValue, double endValue) : this(count, startValue, endValue, false) { }
		
		public ScaleByBounds() : this(128, 0.0, 1.0, false) { }
		
		public int Count {
			get { return count; }
			set { 
				count = value;
				increment  = isLogarithmic ? Math.Pow(endValue / startValue, 1.0 / (count - 1)) : (endValue - startValue) / (count - 1);
			}
		}
		
		public string Name {
			get { return name; }
			set { name = value; }
		}
		
		public double StartValue {
			get { return startValue; }
			set { 
				startValue = value;
				increment  = isLogarithmic ? Math.Pow(endValue / startValue, 1.0 / (count - 1)) : (endValue - startValue) / (count - 1);
			}
		}
		
		public double EndValue {
			get { return endValue; }
			set { 
				endValue = value;
				increment  = isLogarithmic ? Math.Pow(endValue / startValue, 1.0 / (count - 1)) : (endValue - startValue) / (count - 1);
			}
		}
		
		public string Unit {
			get { return unit; }
			set { unit = value; }
		}
		
		/// <summary>
		/// Either the ratio or the difference between successive elements.
		/// </summary>
		/// <returns>The ratio of successive elements if <c>IsLogarithmic</c> is true; the difference between successive elements, otherwise.</returns>
		public double Increment {
			get { return increment; }
		}
		
		public bool IsLogarithmic {
			get { return isLogarithmic; }
			set { 
				isLogarithmic = value;
				increment  = isLogarithmic ? Math.Pow(endValue / startValue, 1.0 / (count - 1)) : (endValue - startValue) / (count - 1);
			}
		}
		
		public double this[int index] {
			get { return isLogarithmic ? startValue * Math.Pow(increment, index) : startValue + index * increment; }
		}
		
		
		public IScale Clone()
		{
			return new ScaleByBounds(count, startValue, endValue, isLogarithmic);
		}
		
		public double[] GetValues()
		{
			double[] values = new double[count];
			
			for (int i = 0; i < count; i++) {
				values[i] = this[i];
			}
						
			return values;
		}
		
		public void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "Scale") throw new ArgumentException("Cannot load Scale data from a different xml element.");
			if (xmlElement.HasChildNodes) {
				int countLocal = -1;
				double startValueLocal = Double.NaN;
				double endValueLocal = Double.NaN;
				bool isLogarithmicLocal = false;
				bool success = true;
				string nameLocal = xmlElement.Attributes["name"].Value;
				string unitLocal = xmlElement.Attributes["unit"].Value;
				foreach (XmlNode child in xmlElement.ChildNodes) {
					if (!(child is XmlElement)) continue;
					switch (child.Name) {
						case "Count":
							success &= Int32.TryParse(child.InnerXml, out countLocal);
							break;
						case "StartValue":
							success &= Double.TryParse(child.InnerXml, out startValueLocal);
							break;
						case "EndValue":
							success &= Double.TryParse(child.InnerXml, out endValueLocal);
							break;
						case "IsLogarithmic":
							success &= Boolean.TryParse(child.InnerXml, out isLogarithmicLocal);
							break;
						default:
							break;
					}
				}
				
				if (success) {
					count = countLocal;
					startValue = startValueLocal;
					endValue = endValueLocal;
					isLogarithmic = isLogarithmicLocal;
					increment  = isLogarithmic ? Math.Pow(endValue / startValue, 1.0 / (count - 1)) : (endValue - startValue) / (count - 1);
					name = nameLocal;
					unit = unitLocal;
				} else {
					count = 0;
					startValue = Double.NaN;
					endValue = Double.NaN;
					isLogarithmic = false;
					increment  = Double.NaN;
					name = String.Empty;
					unit = String.Empty;
				}
			}
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement xml = context.CreateElement("Scale");
			XmlElement child;
			
			xml.SetAttribute("type", "ByBounds");
			
			if (!String.IsNullOrEmpty(name)) {
				xml.SetAttribute("name", name);
			}
			
			if (!String.IsNullOrEmpty(unit)) {
				xml.SetAttribute("unit", unit);
			}
			
			child = context.CreateElement("StartValue");
			child.InnerXml = startValue.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("EndValue");
			child.InnerXml = endValue.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("Count");
			child.InnerXml = count.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("IsLogarithmic");
			child.InnerXml = isLogarithmic.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("Increment");
			child.InnerXml = increment.ToString();
			xml.AppendChild(child);
			
			return xml;
		}
		
	}
}
