﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.05.07.
 * Idő: 15:30
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */

using System;
using System.Xml;

namespace TetheredSun.SignalAnalysis
{
	/// <summary>
	/// Description of ScalingByIncrement.
	/// </summary>
	public class ScaleByIncrement : IScale
	{
		protected int count;
		protected string name = String.Empty;
		protected string unit = String.Empty;
		protected double increment;
		protected double startValue;
		
		public ScaleByIncrement(int count, double increment, double startValue)
		{
			this.count = count;
			this.increment = increment;
			this.startValue = startValue;
		}
		
		public ScaleByIncrement(int count, double increment) : this(count, increment, 0.0) { }
		
		public ScaleByIncrement() : this(128, 1.0, 0.0) { }
		
		public int Count {
			get { return count; }
			set { count = value; }
		}
		
		public string Name {
			get { return name; }
			set { name = value; }
		}
		
		public string Unit {
			get { return unit; }
			set { unit = value; }
		}
		
		
		public double StartValue {
			get { return startValue; }
			set { startValue = value; }
		}
		
		public double EndValue {
			get { return startValue + (count - 1) * increment; }
		}
		
		public double Increment {
			get { return increment; }
			set { increment = value; }
		}
		
		public bool IsLogarithmic {
			get { return false; }
		}
		
		public double this[int index] {
			get { return startValue + index * increment; }
		}
		
		
		public IScale Clone()
		{
			return new ScaleByIncrement(count, increment, startValue);
		}
		
		public double[] GetValues()
		{
			double[] values = new double[count];
			
			for (int i = 0; i < count; i++) {
				values[i] = this[i];
			}
			
			return values;
		}	
	
		public void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "Scale") throw new ArgumentException("Cannot load Scale data from a different xml element.");
			if (xmlElement.HasChildNodes) {
				int countLocal = -1;
				double startValueLocal = Double.NaN;
				double incrementLocal = Double.NaN;
				bool success = true;
				string nameLocal = xmlElement.Attributes["name"].Value;
				string unitLocal = xmlElement.Attributes["unit"].Value;
				foreach (XmlNode child in xmlElement.ChildNodes) {
					if (!(child is XmlElement)) continue;
					switch (child.Name) {
						case "Count":
							success &= Int32.TryParse(child.InnerXml, out countLocal);
							break;
						case "StartValue":
							success &= Double.TryParse(child.InnerXml, out startValueLocal);
							break;
						case "Increment":
							success &= Double.TryParse(child.InnerXml, out incrementLocal);
							break;
						default:
							break;
					}
				}
				
				if (success) {
					count = countLocal;
					increment = incrementLocal;
					startValue = startValueLocal;
					name = nameLocal;
					unit = unitLocal;
				} else {
					count = 0;
					increment = Double.NaN;
					startValue = Double.NaN;
					name = String.Empty;
					unit = String.Empty;
				}
			}
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement xml = context.CreateElement("Scale");
			XmlElement child;
			
			xml.SetAttribute("type", "ByIncrement");
			
			if (!String.IsNullOrEmpty(name)) {
				xml.SetAttribute("name", name);
			}
			
			if (!String.IsNullOrEmpty(unit)) {
				xml.SetAttribute("unit", unit);
			}
			
			child = context.CreateElement("StartValue");
			child.InnerXml = startValue.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("EndValue");
			child.InnerXml = EndValue.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("Count");
			child.InnerXml = count.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("IsLogarithmic");
			child.InnerXml = IsLogarithmic.ToString();
			xml.AppendChild(child);
			
			child = context.CreateElement("Increment");
			child.InnerXml = increment.ToString();
			xml.AppendChild(child);
			
			return xml;
		}
		
	}
}
