﻿/*
 * Készítette a SharpDevelop.
 * Felhasználó: phil
 * Dátum: 2009.05.28.
 * Idő: 11:15
 * 
 * A sablon megváltoztatásához használja az Eszközök | Beállítások | Kódolás | Szabvány Fejlécek Szerkesztését.
 */

using System;
using System.Collections.Generic;
using System.Xml;
using TetheredSun.Core;

namespace TetheredSun.SignalAnalysis
{
	/// <summary>
	/// Description of Signal.
	/// </summary>
	public class Signal : IXmlConvertible
	{
		protected IScale xScale;
		protected IList<double> yValues;
		
		
		public Signal(IScale xScale, IList<double> yValues)
		{
			this.xScale = xScale;
			this.yValues = yValues;
		}
		
		public Signal() : this(null, null) { }
		
		
		public IScale XScale {
			get { return xScale; }
			set { xScale = value; }
		}
		
		public IList<double> YValues {
			get { return yValues; }
			set { yValues = value; }
		}
		
		public double this[int index] {
			get { return yValues[index]; }
			set { yValues[index] = value; }
		}
		
		
		public double GetMean()
		{
			if (yValues.Count == 0) return Double.NaN;
			double mean = 0.0;
			foreach (double y in yValues) {
				mean += y;
			}
			mean = mean / yValues.Count;
			
			return mean;
		}
	
		public Signal Reciprocal()
		{
			Signal result = new Signal();
			result.xScale = xScale.Clone();
			result.yValues = new double[yValues.Count];
			
			for (int i = 0; i < yValues.Count; i++) {
				result.yValues[i] = 1.0 / yValues[i];
			}
			return result;
		}
		
		public Signal SubtractMean()
		{
			return (this - GetMean());
		}
		
		
		public void FromXml(XmlElement xmlElement)
		{
			if (xmlElement.Name != "Signal") throw new ArgumentException("Cannot load Signal data from a different xml element.");
			if (xmlElement.HasChildNodes) {
				foreach (XmlNode child in xmlElement.ChildNodes) {
					if (!(child is XmlElement)) continue;
					switch (child.Name) {
						case "Scale":
							if (child.Attributes["type"].Value == "ByBounds") {
								xScale = new ScaleByBounds();
							} else {
								xScale = new ScaleByIncrement();
							}
							xScale.FromXml(child as XmlElement);
							break;
						case "Data":
							yValues = ConversionTools.GetDoubleList(Convert.FromBase64String(child.InnerXml));
							break;
						default:
							break;
					}
				}
			}
		}
		
		public XmlElement ToXml(XmlDocument context)
		{
			XmlElement xml = context.CreateElement("Signal");
			XmlElement child;
			
			if (xScale == null) return xml;
			if (yValues == null) return xml;
			
			child = xScale.ToXml(context);
			xml.AppendChild(child);
			
			child = context.CreateElement("Data");
			child.SetAttribute("count", yValues.Count.ToString());
			child.SetAttribute("binaryEncoding", "Base64");
			child.InnerXml = Convert.ToBase64String(ConversionTools.GetBytes(yValues), Base64FormattingOptions.InsertLineBreaks);
			xml.AppendChild(child);
			
			return xml;
		}
			
		
		public static Signal Average(IList<Signal> signals)
		{
			if (signals == null) throw new NullReferenceException();
			if (signals.Count < 1) throw new InvalidOperationException("The list of signals is empty.");
			Signal average = new Signal(signals[0].XScale.Clone(), new double[signals[0].YValues.Count]);
			
			for (int i = 0; i < signals.Count; i++) {
				for (int j = 0; (j < average.yValues.Count) && (j < signals[i].yValues.Count); j++) {
					average[j] += signals[i][j];
				}
			}
			
			return average / signals.Count;
		}
		
		public static Signal Normalise(Signal signal)
		{
			double min = Double.MaxValue;
			double max = Double.MinValue;
			double range;
			
			for (int i = 0; i < signal.YValues.Count; i++) {
				if (signal[i] < min) min = signal[i];
				if (signal[i] > max) max = signal[i];
			}
			
			range = max - min;
			return (1.0 / range) * (signal - min);
		}
		
		
		public static Signal operator-(Signal signal)
		{
			Signal result = new Signal();
			result.xScale = signal.xScale.Clone();
			result.yValues = new double[signal.yValues.Count];
			
			for (int i = 0; i < signal.yValues.Count; i++) {
				result.yValues[i] = -signal.yValues[i];
			}
			return result;
		}
		
		public static Signal operator +(Signal signal, double d)
		{
			Signal result = new Signal();
			result.xScale = signal.xScale.Clone();
			result.yValues = new double[signal.yValues.Count];
			
			for (int i = 0; i < signal.yValues.Count; i++) {
				result.yValues[i] = signal[i] + d;
			}
			return result;
		}
	
		public static Signal operator +(double d, Signal signal)
		{
			return signal + d;
		}
	
		public static Signal operator -(Signal signal, double d)
		{
			return signal + (-d);
		}
	
		public static Signal operator -(double d, Signal signal)
		{
			return d + (-signal);
		}
		
		public static Signal operator *(double d, Signal signal)
		{
			Signal result = new Signal();
			result.xScale = signal.xScale.Clone();
			result.yValues = new double[signal.yValues.Count];
			
			for (int i = 0; i < signal.yValues.Count; i++) {
				result.yValues[i] = signal[i] * d;
			}
			return result;
		}
		
		public static Signal operator *(Signal signal, double d)
		{
			return d * signal;
		}
		
		public static Signal operator /(Signal signal, double d)
		{
			Signal result = new Signal();
			result.xScale = signal.xScale.Clone();
			result.yValues = new double[signal.yValues.Count];
			
			for (int i = 0; i < signal.yValues.Count; i++) {
				result.yValues[i] = signal[i] / d;
			}
			return result;
		}
	
		public static Signal operator /(double d, Signal signal)
		{
			return d * signal.Reciprocal();
		}
	
		// TODO: Implement operations between signals.
	}
}
