/*
 * Copyright (C) 1984-2000  Mark Nudelman
 *
 * You may distribute under the terms of either the GNU General Public
 * License or the Less License, as specified in the README file.
 *
 * For more information about less, or for information on how to 
 * contact the author, see the README file.
 */


/*
 * Handling functions for command line options.
 *
 * Most options are handled by the generic code in option.c.
 * But all string options, and a few non-string options, require
 * special handling specific to the particular option.
 * This special processing is done by the "handling functions" in this file.
 *
 * Each handling function is passed a "type" and, if it is a string
 * option, the string which should be "assigned" to the option.
 * The type may be one of:
 *      INIT    The option is being initialized from the command line.
 *      TOGGLE  The option is being changed from within the program.
 *      QUERY   The setting of the option is merely being queried.
 */

#include "less.h"
#include "option.h"

/* extern int nbufs; */
extern int cbufs;
extern int pr_type;
extern int plusoption;
extern int swindow;
extern int sc_height;
extern int secure;
extern int dohelp;
extern int any_display;
extern char openquote;
extern char closequote;
extern char *prproto[];
extern char *eqproto;
extern char *hproto;
extern IFILE curr_ifile;
extern char version[];
#if LOGFILE
extern char *namelogfile;
extern int force_logfile;
extern int logfile;
#endif
#if TAGS
public char *tagoption = NULL;
extern char *tags;
extern int jump_sline;
#endif
#if MSDOS_COMPILER
extern int nm_fg_color, nm_bg_color;
extern int bo_fg_color, bo_bg_color;
extern int ul_fg_color, ul_bg_color;
extern int so_fg_color, so_bg_color;
extern int bl_fg_color, bl_bg_color;
#endif


#if LOGFILE
/*
 * Handler for -o option.
 */
        public void
opt_o(
        int type,
        char *s)
{
        PARG parg;

        if (secure)
        {
                error("log file support is not available", NULL_PARG);
                return;
        }
        switch (type)
        {
        case INIT:
                namelogfile = s;
                break;
        case TOGGLE:
                if (ch_getflags() & CH_CANSEEK)
                {
                        error("Input is not a pipe", NULL_PARG);
                        return;
                }
                if (logfile >= 0)
                {
                        error("Log file is already in use", NULL_PARG);
                        return;
                }
                s = skipsp(s);
                namelogfile = lglob(s);
                use_logfile(namelogfile);
                sync_logfile();
                break;
        case QUERY:
                if (logfile < 0)
                        error("No log file", NULL_PARG);
                else
                {
                        parg.p_string = unquote_file(namelogfile);
                        error("Log file \"%s\"", &parg);
                        free(parg.p_string);
                }
                break;
        }
}

/*
 * Handler for -O option.
 */
        public void
opt__O(
        int type,
        char *s)
{
        force_logfile = TRUE;
        opt_o(type, s);
}
#endif

/*
 * Handlers for -l option.
 */
        public void
opt_l(
        int type,
        char *s)
{
        int err;
        int n;
        char *t;
        char getnumpar;           /* FCS */
  
        switch (type)
        {
        case INIT:
                t = s;
                getnumpar='l';    /* FCS */
                n = getnum(&t, &getnumpar, &err);  /* FCS */
                if (err || n <= 0)
                {
                        error("Line number is required after -l", NULL_PARG);
                        return;
                }
                plusoption = TRUE;
                ungetsc(s);
                break;
        }
}

#if USERFILE
        public void
opt_k(
        int type,
        char *s)
{
        PARG parg;

        switch (type)
        {
        case INIT:
                if (lesskey(s, 0))
                {
                        parg.p_string = unquote_file(s);
                        error("Cannot use lesskey file \"%s\"", &parg);
                        free(parg.p_string);
                }
                break;
        }
}
#endif

#if TAGS
/*
 * Handler for -t option.
 */
        public void
opt_t(
        int type,
        char *s)
{
        IFILE save_ifile;
        POSITION pos;

        switch (type)
        {
        case INIT:
                tagoption = s;
                /* Do the rest in main() */
                break;
        case TOGGLE:
                if (secure)
                {
                        error("tags support is not available", NULL_PARG);
                        break;
                }
                findtag(skipsp(s));
                save_ifile = save_curr_ifile();
                if (edit_tagfile())
                        break;
                if ((pos = tagsearch()) == NULL_POSITION)
                {
                        reedit_ifile(save_ifile);
                        break;
                }
                unsave_ifile(save_ifile);
                jump_loc(pos, jump_sline);
                break;
        }
}

/*
 * Handler for -T option.
 */
        public void
opt__T(
        int type,
        char *s)
{
        PARG parg;

        switch (type)
        {
        case INIT:
                tags = s;
                break;
        case TOGGLE:
                s = skipsp(s);
                tags = lglob(s);
                break;
        case QUERY:
                parg.p_string = unquote_file(tags);
                error("Tags file \"%s\"", &parg);
                free(parg.p_string);
                break;
        }
}
#endif

/*
 * Handler for -p option.
 */
        public void
opt_p(
        int type,
        register char *s)
{
        switch (type)
        {
        case INIT:
                /*
                 * Unget a search command for the specified string.
                 * {{ This won't work if the "/" command is
                 *    changed or invalidated by a .lesskey file. }}
                 */
                plusoption = TRUE;
                ungetsc(s);
                ungetsc("/");
                break;
        }
}

/*
 * Handler for -P option.
 */
        public void
opt__P(
        int type,
        register char *s)
{
        register char **proto;
        PARG parg;

        switch (type)
        {
        case INIT:
        case TOGGLE:
                /*
                 * Figure out which prototype string should be changed.
                 */
                switch (*s)
                {
                case 's':  proto = &prproto[PR_SHORT];  s++;    break;
                case 'm':  proto = &prproto[PR_MEDIUM]; s++;    break;
                case 'M':  proto = &prproto[PR_LONG];   s++;    break;
                case '=':  proto = &eqproto;            s++;    break;
                case 'h':  proto = &hproto;             s++;    break;
                default:   proto = &prproto[PR_SHORT];          break;
                }
                free(*proto);
                *proto = save(s);
                break;
        case QUERY:
                parg.p_string = prproto[pr_type];
                error("%s", &parg);
                break;
        }
}

/*
 * Handler for the -b option.
 */
        /*ARGSUSED*/
        public void
opt_b(
        int type,
        char *s)
{
        switch (type)
        {
        case TOGGLE:
        case QUERY:
                /*
                 * Allocate the new number of buffers.
                 */
                cbufs = ch_nbuf(cbufs);
                break;
        case INIT:
                break;
        }
}

/*
 * Handler for the -i option.
 */
        /*ARGSUSED*/
        public void
opt_i(
        int type,
        char *s)
{
        switch (type)
        {
        case TOGGLE:
                chg_caseless();
                break;
        case QUERY:
        case INIT:
                break;
        }
}

/*
 * Handler for the -V option.
 */
        /*ARGSUSED*/
        public void
opt__V(
        int type,
        char *s)
{
        switch (type)
        {
        case TOGGLE:
        case QUERY:
                dispversion();
                break;
        case INIT:
                /*
                 * Force output to stdout per GNU standard for --version output.
                 */
                any_display = 1;
                putstr("less ");
                putstr(version);
                putstr("\nCopyright (C) 2000 Mark Nudelman\n\n");
                putstr("less comes with NO WARRANTY, to the extent permitted by law.\n");
                putstr("For information about the terms of redistribution,\n");
                putstr("see the file named README in the less distribution.\n");
                quit(QUIT_OK);
                break;
        }
}

#if MSDOS_COMPILER
/*
 * Parse an MSDOS color descriptor.
 */
        static void
colordesc(
        char *s,
        int *fg_color,
        int *bg_color)
{
        int fg, bg;
        int err;
        char getnumpar;       /* FCS */
        getnumpar='D';        /* FCS */
  
        fg = getnum(&s, &getnumpar, &err);  /* FCS */
        if (err)
        {
                error("Missing fg color in -D", NULL_PARG);
                return;
        }
        if (*s != '.')
                bg = 0;
        else
        {
                s++;
                bg = getnum(&s, &getnumpar, &err);  /* FCS */
                if (err)
                {
                        error("Missing fg color in -D", NULL_PARG);
                        return;
                }
        }
        if (*s != '\0')
                error("Extra characters at end of -D option", NULL_PARG);
        *fg_color = fg;
        *bg_color = bg;
}

/*
 * Handler for the -D option.
 */
        /*ARGSUSED*/
        public void
opt_D(
        int type,
        char *s)
{
        switch (type)
        {
        case INIT:
        case TOGGLE:
                switch (*s++)
                {
                case 'n':
                        colordesc(s, &nm_fg_color, &nm_bg_color);
                        break;
                case 'd':
                        colordesc(s, &bo_fg_color, &bo_bg_color);
                        break;
                case 'u':
                        colordesc(s, &ul_fg_color, &ul_bg_color);
                        break;
                case 'k':
                        colordesc(s, &bl_fg_color, &bl_bg_color);
                        break;
                case 's':
                        colordesc(s, &so_fg_color, &so_bg_color);
                        break;
                default:
                        error("-D must be followed by n, d, u, k or s", NULL_PARG);
                        break;
                }
                if (type == TOGGLE)
                {
                        so_enter();
                        so_exit();
                }
                break;
        case QUERY:
                break;
        }
}
#endif

/*
 * Handler for the -" option.
 */
        public void
opt_quote(
        int type,
        register char *s)
{
        char buf[3];
        PARG parg;

        switch (type)
        {
        case INIT:
        case TOGGLE:
                if (s[1] != '\0' && s[2] != '\0')
                {
                        error("-\" must be followed by 1 or 2 chars", NULL_PARG);
                        return;
                }
                openquote = s[0];
                if (s[1] == '\0')
                        closequote = openquote;
                else
                        closequote = s[1];
                break;
        case QUERY:
                buf[0] = openquote;
                buf[1] = closequote;
                buf[2] = '\0';
                parg.p_string = buf;
                error("quotes %s", &parg);
                break;
        }
}

/*
 * "-?" means display a help message.
 * If from the command line, exit immediately.
 */
        /*ARGSUSED*/
        public void
opt_query(
        int type,
        char *s)
{
        switch (type)
        {
        case QUERY:
        case TOGGLE:
                error("Use \"h\" for help", NULL_PARG);
                break;
        case INIT:
                dohelp = 1;
        }
}

/*
 * Get the "screen window" size.
 */
        public int
get_swindow()
{
        if (swindow > 0)
                return (swindow);
        return (sc_height + swindow);
}

