/*
 * idicon.h -- declarations for DICOM handling functions
 * Copyright (c) 1996-98, Laszlo Nyul
 */

/* $Id: idicon.h,v 1.7 1997/05/21 15:18:45 nyul Exp $ */

#ifndef __IDICON_H
#define __IDICON_H

#include <stdio.h>

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef NULL
# define NULL 0
#endif

#ifndef BOOL
# define BOOL short
#endif

#ifndef TRUE
# define TRUE 1
# define FALSE 0
#endif

#ifndef SEEK_SET
# define SEEK_SET 0
# define SEEK_CUR 1
# define SEEK_END 2
#endif

#ifndef PROTO
# if PROTOTYPES
#  define PROTO(ARGS) ARGS
# else
#  define PROTO(ARGS) ()
# endif
#endif

#define IF_FREE_0(PTR) { if (PTR) free(PTR); (PTR) = 0; }

/* dstrbuf.c */

typedef struct {
  long allocstep;
  short number;
  long *usedsize;
  char **chunks;
} DStringBuffer;

/* fixpath.c */

#ifndef DIR_SEPARATOR
# define DIR_SEPARATOR '/'
#endif

extern char *fixpath PROTO((char *path));

/* getfn.c */

#ifndef PATH_SEPARATOR
# define PATH_SEPARATOR ':'
#endif

typedef char* (*getdeffile_func) (char *, char *, int);

extern char *getfilefromenvpath PROTO((char *env, char *deffilename, char *filename, int n));
extern char *getfilefrompath PROTO((char *path, char *deffilename, char *filename, int n));

/* getopt2.c */

typedef struct {
  int opterr1;
  int optind1;
  int optopt1;
  char *optarg1;
  int opt2;
  int optind2;
  int argc2;
  char *argv2buf;
  char **argv2;
} getopt1s;

extern void initgetopt1 PROTO((getopt1s *opt1s));
extern int getopt1 PROTO((getopt1s *opt1s, int argc, char *argv[], char *opts));

/* log.c */

#define LOG_ERROR   1
#define LOG_WARNING 2
#define LOG_INFO    3
#define LOG_ERROR_NULL   -1
#define LOG_WARNING_NULL -2
#define LOG_INFO_NULL    -3

typedef struct {
  FILE *f;
  short mask;
  void *owner;
} log_session;

extern log_session *log_start PROTO((void *owner, char *logfile, char *ident,
        short mask, short append, int argc, char *argv[]));
extern void log_end PROTO((log_session **s, void *owner));
extern void log_mask PROTO((log_session *s, short mask));
extern void log_message PROTO((log_session *s, char *msg, short level));
extern void log_abort PROTO((log_session *s, char *msg));

/* strtoup.c */

extern char *strtoupper PROTO((char *str, long len));
extern char *strtolower PROTO((char *str, long len));

/* swapbyte.c */

extern void swapbytes PROTO((char *ptr, short n));

#ifndef SHORTSTRINGLEN
#define SHORTSTRINGLEN 40
#endif

#ifndef LONGSTRINGLEN
#define LONGSTRINGLEN 256
#endif

#ifndef LONGLONGSTRINGLEN
#define LONGLONGSTRINGLEN 1024
#endif

#define Nuclear_Medicine_Image_Storage_SOP_Class_UID "1.2.840.10008.5.1.4.1.1.20"
#define Secondary_Capture_Image_Storage_SOP_Class_UID "1.2.840.10008.5.1.4.1.1.7"
#define Implicit_VR_Little_Endian_Transfer_Syntax_UID "1.2.840.10008.1.2"
#define Explicit_VR_Little_Endian_Transfer_Syntax_UID "1.2.840.10008.1.2.1"
#define Explicit_VR_Big_Endian_Transfer_Syntax_UID "1.2.840.10008.1.2.2"

typedef char CodecID[65];

typedef enum {
  TS_Default = -1,
  TS_ImplicitVRLittleEndian = 0,
  TS_ExplicitVRLittleEndian = 1,
  TS_ExplicitVRBigEndian = 2,
  TS_Encapsulated = 3,
  TS_na
} E_TransferSyntax;

typedef enum {
  EG_Always = 0,
  EG_Missing = 1,
  EG_Never = 2
} E_ElementGeneration;

typedef struct {
  unsigned short SamplesPerPixel;
  char PhotometricInterpretation[17];
  unsigned short PlanarConfiguration;
  unsigned short Rows;
  unsigned short Columns;
  unsigned short BitsAllocated;
  unsigned short BitsStored;
  unsigned short HighBit;
  unsigned short PixelRepresentation;
  union {
    unsigned short us;
    short ss;
  } PixelPaddingValue;
  unsigned short NumberOfFrames;
  unsigned long decodedframesize;
} S_ImageDescriptor;

typedef struct {
  unsigned short FrameNumber;
  unsigned short FragmentNumber;
  unsigned long Offset;
  unsigned long Size;
} S_EBOT;
 
typedef struct {
  unsigned long allocsize;
  S_ImageDescriptor Image;
  unsigned short TotalNumberOfFragments;
  char *datafilename;
  unsigned long dataoffset;
  unsigned long datalength;
  E_TransferSyntax TransferSyntax;
  CodecID PixelTransferSyntax;
  S_EBOT EBOT[1];
} S_PixelData;

typedef double         T_VR_FD;
typedef float          T_VR_FL;
typedef long           T_VR_SL;
typedef short          T_VR_SS;
typedef unsigned long  T_VR_UL;
typedef unsigned short T_VR_US;

typedef enum {
  EVR_UNKNOWN=-1,
  EVR_AE=0,
  EVR_AS=1,
  EVR_AT=2,
  EVR_CS=3,
  EVR_DA=4,
  EVR_DS=5,
  EVR_DT=6,
  EVR_FL=7,
  EVR_FD=8,
  EVR_IS=9,
  EVR_LO=10,
  EVR_LT=11,
  EVR_OB=12,
  EVR_OW=13,
  EVR_PN=14,
  EVR_SH=15,
  EVR_SL=16,
  EVR_SQ=17,
  EVR_SS=18,
  EVR_ST=19,
  EVR_TM=20,
  EVR_UI=21,
  EVR_UL=22,
  EVR_US=23,
  EVR_RE=24,
  EVR_XS=25,
  EVR_XU=26,
  EVR_OX=27,
  EVR_XX=28,
  EVR_na
} EVR;

#define LF_char '\n'
#define FF_char '\f'
#define CR_char '\r'

#define VR_ok      0x0000 /* VR is correct */
#define VR_length  0x0001 /* length is less than or greater than the
                             permitted minimum/maximum */
#define VR_char    0x0002 /* contains extra characters */
#define VR_format  0x0004 /* does not satisfy the standard VR format */
#define VR_impval  0x0008 /* contains impossible value */
#define VR_unknown 0x0010 /* VR is unknown */
#define VR_VM      0x0020 /* VM is incorrect */
#define VR_nonperm 0x0040 /* contains nonpermitted value */
#define TY_1       0x0080 /* missing type 1 element */
#define TY_10      0x0100 /* type 1 element with 0 length */
#define TY_2       0x0200 /* missing type 2 element */

typedef struct {
  unsigned long tag;
  char *name;
  EVR VR;
  short VM;
  char *cs_type;
  char *permitted;
} S_TagVR;

typedef struct {
  char *PValueField;
  T_VR_US Tag1;
  T_VR_US Tag2;
  T_VR_UL ValueLength;
  EVR XVRType;
  unsigned short flags;
} DicomElement;

struct DOE;
typedef struct DOE {
  DicomElement elem;
  struct DOE *next;
  struct DOE *prev;
  struct DOE *first;
  struct DOE *last;
  struct DOE *parent;
} DicomObjectElement;

#define CodecLibraryFileName "codeclib.map"

#define UIDDefFileName "uid_tab.def"
#define UID2DefFileName "uid2_tab.def"
#define ABDefFileName "ab_tab.def"
#define CDefFileName "c_tab.def"
#define TagDefFileName "tag_tab.def"
/*#define PrvDefFileName "prv_tab.def"*/
#define AB2DefFileName "ab2_tab.def"
#define C2DefFileName "c2_tab.def"
#define Tag2DefFileName "tag2_tab.def"
/*#define Prv2DefFileName "prv2_tab.def"*/

#define OverlayPlaneTableID "C.9.2"
#define MultiFrameOverlayTableID "C.9.3"
#define CurveTableID "C.10.2"
#define PrivateTableID "P.R.I.V.A.T.E"
#define PrivateTableName "Private Data"

typedef struct {
  unsigned long AttrTag;
  char *AttrName;
  char Type[3];
  char *Description;
  unsigned short flags;
} CTableRowElem;

typedef struct CTR {
  CTableRowElem elem;
  DicomObjectElement *value;
  struct CTR *next;
  struct CTR *prev;
  struct CTR *first;
  struct CTR *last;
  short Cparent;
  union {
    struct ABTR *AB;
    struct CTR *C;
  } parent;
} CTableRow;

typedef struct {
  char *CTableID;
  char *CTableName;
  char Usage[2];
  char *Explanation;
} ABTableRowElem;

typedef struct ABTR {
  ABTableRowElem elem;
  struct ABTR *next;
  struct ABTR *prev;
  struct CTR *first;
  struct CTR *last;
  struct ABT *parent;
} ABTableRow;

typedef struct ABT {
  char *TableID;
  char *TableName;
  struct ABTR *first;
  struct ABTR *last;
  DStringBuffer *DSB;
} ABTable;

/*
typedef struct PRVTR {
  S_TagVR *elem;
  unsigned short flags;
  DicomObjectElement *value;
  struct PRVTR *next;
  struct PRVTR *prev;
  struct PRVTB *parent;
} PrvTableRow;

typedef struct PRVTB {
  char *PrivateCreatorIC;
  struct PRVTB *next;
  struct PRVTB *prev;
  struct PRVTR *first;
  struct PRVTR *last;
  struct PRVT *parent;
} PrvTableBlock;

typedef struct PRVT {
  struct PRVTB *first;
  struct PRVTB *last;
  DStringBuffer *DSB;
} PrvTable;
*/

typedef struct {
  char preamble[128];
  DicomObjectElement *root;
  DicomObjectElement *current;
  DicomObjectElement *DE_current;
  ABTable *AB;
} DicomObject;

/* codec.c */

typedef struct {
  unsigned short FrameNumber;
  unsigned long allocatedsize;
  unsigned long datasize;
  unsigned long skip;
  char *data;
} S_Frame;

typedef struct {
  unsigned short FrameNumber;
  unsigned short FragmentNumber;
  unsigned long offset;
  unsigned long allocatedsize;
  unsigned long datasize;
  unsigned long skip;
  char *data;
} S_CodedFragment;

typedef struct {
  unsigned short FrameNumber;
  unsigned short NumberOfFragments;
  S_CodedFragment **fragments;
} S_CodedFrame;

typedef short (*codeclibraryinit_func) (void *app_private, char *cmdline);
typedef short (*codeclibrarydone_func) (void *app_private);

typedef struct {
  CodecID name;
  void *e_private;
  codeclibrarydone_func done;
} S_CodecLibrary;

typedef short (*codecinit_func) (void *kernel_private,
                                 S_ImageDescriptor *image);
typedef short (*codecdone_func) (void *kernel_private);
typedef short (*coder_func) (void *kernel_private);
typedef short (*decoder_func) (void *kernel_private, S_Frame *frame);

typedef struct {
  CodecID libraryname;
  CodecID name;
  CodecID ID;
  short isCoder; /* true = coder, false = decoder */
  codecinit_func init;
  codecdone_func done;
  union {
    coder_func coder;
    decoder_func decoder;
  } codec;
} S_Codec;

/* a function with wired-in name is called by the kernel to
   initialize codec library */

/* called by the codeclibraryinit function once */
short RegisterCodecLibrary PROTO((void *app_private, S_CodecLibrary *entry));

/* called by the codeclibraryinit function for each codec function
   in the library */
short RegisterCodec PROTO((void *app_private, S_Codec *entry));

/* called by the codeclibraryinit, coderinit and the
   decoderinit functions for each attribute they need */
short GetAttribute PROTO((void *kernel_private, DicomElement *Element));

/* called by coderinit function to set its private data that will
   be needed during the session */
void SetCoderPrivate PROTO((void *kernel_private, void *e_private));

/* called by decoderinit function to set its private data that will
   be needed during the session */
void SetDecoderPrivate PROTO((void *kernel_private, void *e_private));

/* called by coder function to get its private data during the session */
/* called by coderdone function to clean up its private data */
void *GetCoderPrivate PROTO((void *kernel_private));

/* called by decoder function to get its private data during the session */
/* called by decoderdone function to clean up its private data */
void *GetDecoderPrivate PROTO((void *kernel_private));

/* called by coderinit function to get its libraries private data
   if used to set up its own private data */
void *GetCoderLibraryPrivate PROTO((void *kernel_private));

/* called by decoderinit function to get its libraries private data
   if used to set up its own private data */
void *GetDecoderLibraryPrivate PROTO((void *kernel_private));

/* called by the decoder function to get the frame to decode */
short GetCodedFrame PROTO((void *kernel_private, S_Frame **frame,
  short FrameNumber));

/* called by the decoder function to get the frame fragment to decode */
short GetCodedFragment PROTO((void *kernel_private, S_CodedFragment **fragment,
  short FrameNumber, short FragmentNumber));

/* called by the coder function to get the frame to code */
short GetDecodedFrame PROTO((void *kernel_private, S_Frame **frame,
  short FrameNumber));

/* called by the coder function to put the coded frame fragment */
short PutCodedFragment PROTO((void *kernel_private, S_CodedFragment *fragment));

typedef struct {
  DicomObject *O;
  E_TransferSyntax DetectedMetaHeaderTransferSyntax;
  E_TransferSyntax DetectedObjectTransferSyntax;
  E_TransferSyntax MetaHeaderTransferSyntax;
  E_TransferSyntax ObjectTransferSyntax;
  CodecID PixelTransferSyntax;
  E_ElementGeneration MetaHeaderElementGeneration;
  short WithMetaHeader;
  short CopyPixelData;
  short BatchMode;
  short isMetaHeader;
  E_TransferSyntax TransferSyntax;
  E_TransferSyntax newTransferSyntax;
  FILE *fileout;
} DicomIOObject;

#define OECompTag(OE) (unsigned long)(((unsigned long)((OE)->elem.Tag1)<<16)+(unsigned long)((OE)->elem.Tag2))
#define ECompTag(E) (unsigned long)(((unsigned long)((E)->Tag1)<<16)+(unsigned long)((E)->Tag2))
#define CompTag(t1, t2) (unsigned long)(((unsigned long)(t1)<<16)+(unsigned long)(t2))

typedef struct {
  unsigned short NumberOfEntries;
  S_CodecLibrary **entries;
} S_CodecLibRegistry;
 
typedef struct {
  unsigned short NumberOfEntries;
  S_Codec **entries;
} S_CodecRegistry;
 
typedef struct {
  char *title;
  char *version;
  DStringBuffer *TagDSB;
  S_TagVR *TagVR;
  S_TagVR *TagVR2;
  short DIM_OF_TagVR;
  short DIM_OF_TagVR2;
  short Alloc_DIM_OF_TagVR;
  short Alloc_DIM_OF_TagVR2;
/*   PrvTable *Prv; */
  getdeffile_func getdeffile;
  log_session *log;
  unsigned short DIM_OF_TempNames;
  char **TempNames;
  S_CodecLibRegistry codeclibreg;
  S_CodecRegistry codecreg;
} S_AppData;

/* dicomrd.c */

char *InsertTempName(S_AppData *App, char *e_template);
short RemoveTempName(S_AppData *App, char *tempname);

short UnregisterCodecLibrary PROTO((S_AppData *App, S_CodecLibrary *entry));
short UnregisterCodec PROTO((S_AppData *App, S_Codec *entry));
void done_CodecLibraryRegistry PROTO((S_AppData *App));
void init_AppData PROTO((S_AppData *App, char *title, char *version,
        char *version_string));
void done_AppData PROTO((S_AppData *App));
void D_Alloc PROTO((DicomElement *Element));
void D_Free PROTO((DicomElement *Element));
void change_TransferSyntax PROTO((EVR VR, DicomElement *Element));
int bigendian PROTO((void));
char *padwithspace PROTO((char *str));

void init_IOObject PROTO((DicomIOObject *Object));
short DO_openObject PROTO((DicomIOObject *Object));
short DO_importObjectFromFile PROTO((S_AppData *App, char *FileName,
        DicomIOObject *Object));
void DOU_defXVRTypeByTags PROTO((S_AppData *App, T_VR_US Tag1, T_VR_US Tag2,
        EVR *XVRType));
short DO_searchForElement PROTO((DicomObject *O, DicomElement *Element));
/*short DO_searchForPrivateElement PROTO((DicomObject *O, char *Creator,
        DicomElement *Element));*/
short DO_addElement PROTO((S_AppData *App, DicomObject *O,
        DicomElement *Element));
short DO_deleteElement PROTO((S_AppData *App, DicomObject *O,
        DicomElement *Element));
short DO_closeObject PROTO((DicomIOObject *Object));
void SortCodes PROTO((DicomObjectElement *ObjectElement));
void print_element_value PROTO((S_AppData *App,
        DicomObjectElement *ObjectElement, char *buffer, long bufferlen));

void add_Item PROTO((S_AppData *App, DicomObject *O, T_VR_UL length));
void add_Item_Delimitation_Item PROTO((S_AppData *App, DicomObject *O));
void add_Sequence_Delimitation_Item PROTO((S_AppData *App, DicomObject *O));

EVR changeVRnameToVR PROTO((char *VRtext));
char *changeVRToVRname PROTO((EVR VR));
void getVM PROTO((S_AppData *App, unsigned long tag, short *VM));
void getVR PROTO((S_AppData *App, unsigned long tag, EVR *VR));
void getVRtext PROTO((S_AppData *App, unsigned long tag, EVR *VR,
        char **VRtext));
void getCStype PROTO((S_AppData *App, unsigned long tag,
        char **cs_type));
void getPermitted PROTO((S_AppData *App, unsigned long tag,
        char **permitted));
void getAttrName PROTO((S_AppData *App, unsigned long tag,
        char **name));
void getVRandName PROTO((S_AppData *App, unsigned long tag, EVR *VR,
        char **VRtext, char **name));

/* dicomver.c */

short VerifyObjectByModule PROTO((S_AppData *App, DicomObject *O));
short VerifyObjectByTag PROTO((S_AppData *App, DicomObject *O));
short isSQEmpty PROTO((CTableRow *SQ));

/* dicomwr.c */

short DO_generateMetaHeader PROTO((S_AppData *App,
        DicomIOObject *Object));
short DO_checkObject PROTO((S_AppData *App, DicomObject *O));
short DO_calcElementsLength PROTO((S_AppData *App,
        DicomIOObject *Object));

short DO_copyObjectToFile PROTO((S_AppData *App, DicomIOObject *Object,
        char *FileName));
short DE_search PROTO((S_AppData *App, DicomObject *O,
        DicomObjectElement **ObjectElement, T_VR_US Tag1, T_VR_US Tag2,
        short Index));
short DE_add PROTO((S_AppData *App, DicomObject *O,
        DicomElement *Element));
short DE_delete PROTO((S_AppData *App, DicomObject *O,
        DicomElement *Element));

short DEF_search PROTO((S_AppData *App, DicomObject *O, CTableRow *CRow,
        T_VR_US Tag1, T_VR_US Tag2, short Index));
short DEF_add PROTO((S_AppData *App, DicomObject *O, CTableRow *CRow,
        DicomElement *Element));
short DEF_delete PROTO((S_AppData *App, DicomObject *O, CTableRow *CRow));

/* dicomdef.c */

short InitUIDDef PROTO((S_AppData *App, char *classuid, BOOL only_primary,
        char *tableid));
short InitABDef PROTO((S_AppData *App, DicomObject *O, BOOL only_primary,
        char *tableid));
short InitCDef PROTO((S_AppData *App, DicomObject *O, BOOL only_primary));
short InitUABCDef PROTO((S_AppData *App, DicomObject *O, BOOL only_primary));
short InitTagDef PROTO((S_AppData *App, BOOL only_primary));
/* short InitPrvDef PROTO((S_AppData *App, BOOL only_primary)); */
/* short InitTagPrvDef PROTO((S_AppData *App, BOOL only_primary)); */
void DoneABCDef PROTO((DicomObject *O));
void DoneTagDef PROTO((S_AppData *App));
/* void DonePrvDef PROTO((S_AppData *App)); */
/* void DoneTagPrvDef PROTO((S_AppData *App)); */
short BindABCDef PROTO((S_AppData *App, DicomObject *O));
short RebindABCDef PROTO((S_AppData *App, DicomObject *O));
/* short BindPrvDef PROTO((S_AppData *App, DicomObject *O)); */
/* short RebindPrvDef PROTO((S_AppData *App, DicomObject *O)); */
char *getDICOMdeffilefromenvpath PROTO((char *deffilename,
        char *filename, int n));

/* genuid.c */

#define BaseUIDFileName "base.uid"

#define UID_STUDYPART  0x0001
#define UID_SERIESPART 0x0002
#define UID_IMAGEPART  0x0004

#define UID_STUDY      (UID_STUDYPART)
#define UID_SERIES     (UID_STUDYPART | UID_SERIESPART)
#define UID_IMAGE      (UID_STUDYPART | UID_SERIESPART | UID_IMAGEPART)

short gen_stamp PROTO((S_AppData *App, DicomObject *O, char *stamp));
short genUID PROTO((S_AppData *App, DicomObject *O, short level, char *UID,
        char *stamp));
short genUID2 PROTO((S_AppData *App, DicomObject *O, short level, char *UID,
        char *stamp, char *prefix, char *study, char *series,
        char *acquisition, char *image));

/* interfil.c */

#define INTERFILEDefFileName "interfil.def"

#define MAXKEYS 150
#define BUFFERALLOCSTEP 1024
#define MAXASCIIlist 8

typedef enum {
  EVT_UNKNOWN = 0,
  EVT_Numeric = 1,
  EVT_ASCII = 2,
  EVT_ASCIIlist = 3,
  EVT_DateFormat = 4,
  EVT_TimeFormat = 5,
  EVT_NULL = 6
} EVT;

extern char valuetypenames[][11];

typedef enum {
  EKT_UNKNOWN = 0,
  EKT_SIMPLE = 1,
  EKT_INDEXED = 2
} EKT;

typedef enum {
  ECT_UNKNOWN = 0,
  ECT_KVP = 1,
  ECT_IF = 2,
  ECT_FOR = 3
} ECT;

typedef struct {
  ECT typeofcell;
  short keykey;
  char *valueptr;
} InterfileCell;

typedef struct IFE {
  InterfileCell cell;
  struct IFE *next;
  struct IFE *prev;
  struct IFE *first;
  struct IFE *last;
  struct IFE *parent;
} InterfileElement;

typedef struct {
  char *filename;
  DStringBuffer *DSB;
  short usedkey;
  char *keys[MAXKEYS];
  char *normkeys[MAXKEYS];
  BOOL obligatory[MAXKEYS];
  EKT typeofkey[MAXKEYS];
  EVT typeofvalue[MAXKEYS];
  char *defaults[MAXKEYS];
  char *values[MAXKEYS][MAXASCIIlist];
  short newloop;
  InterfileElement *root;
  InterfileElement *current;
} INTERFILEdef;

typedef struct {
  char *filename;
  DStringBuffer *DSB;
  InterfileElement *root;
  InterfileElement *current;
  INTERFILEdef *Idef;
} INTERFILE;

typedef struct {
  char *title;
  char *version;
  getdeffile_func getdeffile;
  log_session *log;
  INTERFILEdef *Idef;
} S_IAppData;

short InitINTERFILEdef PROTO((S_IAppData *IApp));
void DoneINTERFILEdef PROTO((S_IAppData *IApp));

char *getINTERFILEdeffilefromenvpath PROTO((char *deffilename,
        char *filename, int n));
void init_IAppData PROTO((S_IAppData *IApp, char *title, char *version,
        char *version_string));
void done_IAppData PROTO((S_IAppData *IApp));

char *getInterfileName PROTO((INTERFILE *I));
short initinterfile PROTO((S_IAppData *IApp, INTERFILE **I));
void doneinterfile PROTO((INTERFILE **I));
short readinterfileheader PROTO((S_IAppData *IApp, INTERFILE **I,
        char *filename));

short findinterkey PROTO((INTERFILE *I, char *key));
char *getinterkey PROTO((INTERFILE *I, short keykey));
short getintervalue PROTO((INTERFILE *I, short keykey, short index,
        char *buffer));
short getintervaluedef PROTO((INTERFILE *I, short keykey, short index,
        char *buffer));
char *getinterdefaultvalue PROTO((INTERFILE *I, short keykey));
short putintervalue PROTO((INTERFILE *I, short keykey, short index,
        char *buffer, BOOL add));

EKT getinterkeytype PROTO((INTERFILE *I, short keykey));
EVT getintervaluetype PROTO((INTERFILE *I, short keykey));
BOOL obligatoryinterkey PROTO((INTERFILE *I, short keykey));

short checkinterfileheader PROTO((INTERFILE *I));
short native_LibraryInit(void *app_private, char *cmdline) ;
short png_LibraryInit(void *app_private, char *cmdline) ;

#ifdef __cplusplus
}
#endif

#endif /* !__IDICON_H */
