// levenberg-marquardt in java 
//
// To use this, implement the functions in the LMfunc interface.
//
// This library uses simple matrix routines from the JAMA java matrix package,
// which is in the public domain.  Reference:
//    http://math.nist.gov/javanumerics/jama/
// (JAMA has a matrix object class.  An earlier library JNL, which is no longer
// available, represented matrices as low-level arrays.  Several years 
// ago the performance of JNL matrix code was better than that of JAMA,
// though improvements in java compilers may have fixed this by now.)
//
// One further recommendation would be to use an inverse based
// on Choleski decomposition, which is easy to implement and
// suitable for the symmetric inverse required here.  There is a choleski
// routine at idiom.com/~zilla.
//
// If you make an improved version, please consider adding your
// name to it ("modified by ...") and send it back to me
// (and put it on the web).
//
// ----------------------------------------------------------------
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA  02111-1307, USA.
//
// initial author contact info:  
// jplewis  www.idiom.com/~zilla  zilla # computer.org,   #=at
//
// Improvements by:
// dscherba  www.ncsa.uiuc.edu/~dscherba  
// Jonathan Jackson   j.jackson # ucl.ac.uk


package levenbergMarquard;

import Jama.*;

public final class LM
{
	public static int npts;
	public static int nparm;
	public static double e0;
	public static int iter;
	public static double e1;
	
	
	public static double delta = 1e-6;
	public static double idelta = 1.0 / delta;

	
  public static double chiSquared(double[] a, double[] y, LMfunc f)
  {
    double d, sum = 0.;
    
    for( int i = 0; i < y.length; i++ ) {
      d = y[i] - f.val(i, a);
      sum += d*d;
    }
    return sum;
  }


  /**
   * Minimize E = sum {(y[k] - f(x[k],a))}^2
   * Note that LMfunc implements the value and gradient of f(x,a),
   * NOT the value and gradient of E with respect to a!
   * 
   * @param a the parameters/state of the model
   * @param y corresponding array of values
   * @param lambda blend between steepest descent (lambda high) and
   *	jump to bottom of quadratic (lambda zero).
   * 	Start with 0.001.
   * @param termepsilon termination accuracy
   * @param maxiter	stop and return after this many iterations if not done
   *
   * @return the new lambda for future iterations.
   */
  public static void solve(double[] a, double[] y, LMfunc f,
			     double lambda, double termepsilon, int maxiter)
    throws Exception
  {
    npts = y.length;		//38
    nparm = a.length;		//12 v 6
    
    double[] d, na;
    double puffer;
    int c0;
    
    e0 = chiSquared(a, y, f);
    
    boolean done = false;

    double[][] H = new double[nparm][nparm];
    double[] g = new double[nparm];
    
    double[][] grad = new double[npts][nparm];
    double[] fValues = new double[npts];

    iter = 0;
    int term = 0;	// termination count test

    do {
      ++iter;
      
      //store function values
      for( int i = 0; i < npts; i++ ) {
    	  fValues[i] = f.val(i, a);
      }
      
      //gradient (numerical)
      for( int c = 0; c < nparm; c++ ) {
		  a[c] += delta;
		  for( int i = 0; i < npts; i++ ) {
			  grad[i][c] = (f.val(i, a) - fValues[i]) * idelta;
		  }
		  a[c] -= delta;
      }

      // hessian approximation
      //diagonal elements
      for( int c = 0; c < nparm; c++ ) {
		  for( int i = 0; i < npts; i++ ) {
			  
			  if (i == 0) H[c][c] = 0.;
			  
			  puffer = grad[i][c];
			  H[c][c] += (puffer * puffer);
			
		  }  //npts
	  } //c
      
      //upper triangle part
      c0 = 1;
      for( int r = 0; r < nparm; r++ ) {
    	  for( int c = c0; c < nparm; c++ ) {
    		  for( int i = 0; i < npts; i++ ) {
    			  
    			  if (i == 0) H[r][c] = 0.;
        			  
    			  H[r][c] += (grad[i][r] * grad[i][c]);
    			  
    		  }  //npts
    	  } //c
    	  c0++;
      } //r
      
      //lower triangle part
      c0 = 1;
      for( int r = 0; r < nparm; r++ ) {
    	  for( int c = c0; c < nparm; c++ ) {
    		  H[c][r] = H[r][c];
    	  }
    	  c0++;
      }
      
       // boost diagonal towards gradient descent
      for( int r = 0; r < nparm; r++ )
    	  H[r][r] *= (1. + lambda);

      // gradient
      for( int r = 0; r < nparm; r++ ) {
    	  for( int i = 0; i < npts; i++ ) {
    		  
    		  if (i == 0) g[r] = 0.;
    		  
    		  g[r] += ((y[i] - fValues[i]) * grad[i][r]);
    	  }
      } //npts


      d = (new Matrix(H)).lu().solve(new Matrix(g, nparm)).getRowPackedCopy();
      na = (new Matrix(a, nparm)).plus(new Matrix(d, nparm)).getRowPackedCopy();
      e1 = chiSquared(na, y, f);


      // termination test (slightly different than NR)
      if (Math.abs(e1-e0) > termepsilon) {
    	  term = 0;
      } else {
    	  term++;
		if (term == 4) {
		  done = true;
		}
      }
      
      if (iter >= maxiter) done = true;


      if (e1 > e0 || Double.isNaN(e1)) { // new location worse than before
    	  lambda = Math.min(lambda * 11, 1.e7);
      } else {		// new location better, accept new parameters
    	  lambda = Math.max(lambda / 9, 1.e-7);
    	  e0 = e1;
	
		for( int i = 0; i < nparm; i++ ) {
		  a[i] = na[i];
		}
      }

    } while(!done);

  } //solve
  
} //LM

